%%        MATLAB code that calculates the best-fitted ellipsoid           |
% Input: 3 dimensional matrix (D) where the inclusion is 1 and host is 0  |
%        spatial increment in x y z direction as dx dy and dz             |
% Output: orientation and axes length of the best-fitted ellipsoid        |
% The orientation is described by directional angle, alpha, beta and gamma|
% They are the angles of the vector with respect to x, y and z axes       |
% Example: ellipsoid fit                                                  |
% Developer: Xin Zhong, Freie University Berlin, Germany                  |
% Contact: xinzhong0708@gmail.com                                         |
%==========================================================================

%% Start
clear;figure(1);clf;hold on;clc;addpath([cd,'/bin']);format short

%% Example for ellipsoid
%Ellipsoid axes (example, the result should give these three values)
a          =  2.0;
b          =  1.0;
c          =  3.0;
%Coordinate (larger than the inclusion, should not truncate the inclusion)
x1         =  linspace(-3,3,100); dx=x1(2)-x1(1);
y1         =  linspace(-3,3,100); dy=y1(2)-y1(1);
z1         =  linspace(-3,3,100); dz=z1(2)-z1(1);
[x3,y3,z3] =  ndgrid(x1,y1,z1);
r          =  x3.^2/a^2 + y3.^2/b^2 + z3.^2/c^2;
D          =  zeros(size(x3));
D(r<1)     =  1;

%% Function for fitting the ellipsoid.
%Input: dx dy dz and D (3D matrix, 1 for inclusion, 0 for surroundings)
[A,B]      =  Fit_Ellip(dx,dy,dz,D);
%Output: A: axes' length
%        B: Orientation of maximal and minimal axes
%If the output shows: "Optimization completed because the objective function is non-decreasing in feasible directions". This means that convergence is reached. 
%Otherwise, please contact: xinzhong0708@gmail.com. Initial guess in "Fit_Ellip" needs to be changed as this is a non-linear optimization

%% Display result
disp(['The longest axes length: ' ,num2str(A(1))])
disp(['The middle axes length: '  ,num2str(A(2))])
disp(['The shortest axes length: ',num2str(A(3))])
disp(['The direction angle of longest axis: ' ,num2str(real(acos(B(1,:))/pi*180))])
disp(['The direction angle of shortest axis: ',num2str(real(acos(B(3,:))/pi*180))])