"""This is the Python code used to generate the figures 7a and b and explained in Appendix C
of the manuscript entitled: An evaluation of different measures of dynamically recrystallized
grain size for paleopiezometry or paleowattometry studies by Lopez-Sanchez and Llana-Funez

Requirements: Python 2.7.x or higher; Numpy 1.5 or higher; Matplotlib 1.3 or higher"""

### a Monte Carlo simulation to find the minimum sample size needed to reach reproducibility

from __future__ import division, print_function # this is to avoid python 2.x - 3.x compatibility issues
import numpy
import matplotlib.pyplot as plt

def sampleGenerator(sampleSize):
    """Generates a sub-sample of defined size by ramdomly choosing grains 
    from natural sample MAL05.
    
    INPUT
    sampleSize: the size of the sample. An integer
    """
    
    # Generates a random sample of defined size from the sample MAL05
    dataset = numpy.random.choice(sampleMAL05, size=sampleSize)
        
    return dataset

def getGrainSize(sampleSize, numTrials):
    """Calculate the average grain size. Returns a numpy array
    with the average grain size estimated in each run. 
    
    INPUTS
    sampleSize: the sample size defined. An integer.
    numTrials: the number of runs per sample
    """
    
    aGSList = numpy.array([])
    
    for i in range(numTrials):
        
        # generates the sample
        sample = sampleGenerator(sampleSize)
                
        # calculates the average grain size
        aveGrainSize = numpy.mean(sample)
        
        # store the result
        aGSList = numpy.append(aGSList, aveGrainSize)
    
    return aGSList

def findSampleSize(sampleSize=1, step=1, numTrials=5000):
    """Calculates the mean, the standard deviation and the coefficient of variation
    of each simulation. It returns several python lists to make the plots including:
    (1) a list with the sample size; (2) the means, (3) the standard deviations, and
    (4) the 2-sigma or 3-sigma coefficient of variation
    
    INPUTS
    sampleSize: the initial sample size, an integer.
    step: spacing between two adjacent values, an integer.
    numTrials: the number of trials per sample, an integer.
    """
    
    # Initialize variables
    meanList = []
    stdDevList = []
    sizeList = []
    coeffVarList = []
    coeffVarList3 = []
    coeffVar_2sig = 1
    coeffVar_3sig = 1
    
    while coeffVar_3sig > 0.04:
        
        # get the average grain sizes for the current sample size
        aGSList = getGrainSize(sampleSize, numTrials)
                    
        # calculates the mean and the standard deviation to check REPRODUCIBILITY
        mean = numpy.mean(aGSList)
        stdDev = numpy.std(aGSList)
        coeffVar_2sig = (2*stdDev)/float(mean) # coefficient of variation (2-sigma)
        coeffVar_3sig = (3*stdDev)/float(mean)
        print ('sample size: ', sampleSize)
        print ('mean: ', mean)
        print ('stdDev: ', stdDev)
        print ('coeffVar(2-sigma): ', coeffVar_2sig)
        print ('coeffVar(3-sigma): ', coeffVar_3sig)
        print (' ')
        
        # store all the data
        meanList.append(mean)
        stdDevList.append(stdDev)
        coeffVarList.append(coeffVar_2sig)
        coeffVarList3.append(coeffVar_3sig)
        sizeList.append(sampleSize)
        
        # update the sample size
        sampleSize += step # same as sampleSize = sampleSize + step
    
    
    print ('sizeList = ', sizeList)

    return generatePlot(sizeList, meanList, stdDevList, coeffVarList, coeffVarList3, sampleSize, step)

def generatePlot(sizeList, meanList, stdDevList,coeffVarList, coeffVarList3, sampleSize, step):
    """Return a plot containing the two subplots showed in figure 10c, d"""

    plt.figure(figsize=(15,5.63))

    # make the mean/std plot
    plt.subplot(121) #create a subplot
    plt.semilogy(sizeList, meanList, 'o-', color='#66c2a5', label='mean values', linewidth=1.5)
    plt.semilogy(sizeList, stdDevList, '-', color='#8da0cb', label='standard deviation', linewidth=1.5)
    plt.xlabel('Sample Size')
    plt.ylabel('Mean')
    plt.legend()

    # make the coefficient of variation plot
    plt.subplot(122)
    plt.semilogy(sizeList, coeffVarList, '-', color='#ca0020', label='2-sigma', linewidth=1.5)
    plt.semilogy(sizeList, coeffVarList3, '-', color='#0571b0', label='3-sigma', linewidth=1.5)
    plt.plot([0, 1000], [0.04, 0.04], '--', color='#bababa')
    plt.plot([sampleSize-step, sampleSize-step], [0, 1], '--', color='#bababa')
    plt.xlabel('Sample Size')
    plt.ylabel('Coefficient of Variation (2-sigma)')
    plt.legend()

    return plt.show()


# The numpy array with all the diameters of the grains measured in the sample MAL05

sampleMAL05 = numpy.array([43.95009562812275, 42.888942443847505, 82.13642931480233, 45.850247031137606, 48.31837374160721, 41.00771931506743, 39.297909200034354, 34.12852188016045, 78.5392950266545, 33.97357762664988, 32.91303591228398, 55.40939337537207, 39.030766365135385, 51.78410384319464, 24.261178062403662, 23.51449326335812, 55.79270261639163, 49.422159395803455, 35.10724891741041, 37.96207066708141, 40.807269588890705, 26.21755088998049, 37.784149490461175, 47.65569408171564, 59.20101274212229, 31.042970871948622, 64.344085329222, 23.51449326335812, 28.832558362274582, 47.76422667843579, 48.026741785655894, 74.29567055874037, 23.97203696599851, 60.764772837703, 33.164503520930516, 56.57719083322885, 63.25528348219927, 32.774998827447256, 69.3567786613502, 23.046922653056157, 16.02850099583284, 40.55061747623467, 25.89276930120869, 27.475175605661153, 34.54535680432343, 38.529475519763935, 48.13439940326619, 38.27621579026203, 57.081107880999475, 14.06070206025333, 31.527647768723536, 22.6376951158032, 28.566653738546428, 49.91275497049215, 53.587456514871754, 74.01106958869048, 47.12480561077685, 84.641660938458, 23.046922653056157, 49.45203514555295, 55.07672387662922, 44.653890131193734, 32.357150189531666, 63.32462701384879, 37.18453701356991, 50.28110003652069, 27.72482843969876, 45.01794389494325, 29.614852673240364, 82.66477456948459, 10.928916663120132, 22.079027850573283, 48.68415596561005, 32.958904136813565, 38.11949464339053, 55.85851495057194, 44.4873759111068, 24.133145994489517, 44.47069750625689, 39.04991822030895, 50.38373036082104, 34.39241327414974, 50.00142147707695, 30.871408853814103, 62.45216888549379, 27.80748230700263, 39.69503802488848, 28.593349919071766, 29.89616229088037, 36.265386895613645, 32.59000480596934, 31.334737242440916, 45.672071149417505, 35.085807298432314, 27.641882066000203, 41.423562530562975, 31.06740617892473, 48.60819033808796, 56.874938676073434, 88.40243069748385, 44.52074306563912, 38.72309595648138, 50.25173878974073, 52.47670542540407, 32.98182300377667, 46.26850996352991, 58.828268139724464, 58.062703728882084, 84.16914354530567, 54.78223914384664, 68.16873194602813, 20.684917087042766, 12.87813131882624, 25.168228803826167, 34.82750912845684, 27.779968855143885, 23.842301540448304, 78.30768313768712, 54.91630052191119, 21.901204463221966, 53.035103197786434, 57.73319896156483, 17.97506726745811, 61.08933313993106, 44.21959267958853, 35.638624469810416, 14.694640400560834, 45.63960116745909, 37.8039523976041, 23.71175749673423, 57.18390884093351, 33.346095291295825, 42.13645773910371, 36.63598710721906, 21.467771325552214, 48.36425643124222, 36.51289111981349, 58.26452360479963, 35.638624469810416, 15.244449832225811, 81.30159218859589, 34.654164918087616, 43.47423179569309, 27.530857549163926, 57.52944420515134, 27.110207483112738, 42.802159535304995, 68.06161459686463, 26.882921524190767, 34.370516283970765, 48.790297429748044, 27.586442781583266, 54.1611170658679, 26.011377485246076, 20.144168024862445, 32.774998827447256, 35.15007114318006, 75.3265323744566, 71.57032150233107, 85.00068292531294, 57.41450946132924, 42.50570626987618, 42.888942443847505, 90.89809058632498, 71.51940832491863, 20.416539284357984, 42.888942443847505, 15.027354308740744, 49.03200295810208, 51.45564784956998, 22.80919239872138, 43.182629687460086, 66.95892360175124, 49.13735933284912, 21.613308456517963, 53.669791740532496, 36.40997985728019, 37.20467826624061, 65.17916855540315, 51.812559870225414, 24.261178062403662, 26.538040802140245, 24.890603291396186, 34.54535680432343, 63.20901768490879, 52.1669271291488, 49.466959231114345, 14.581780505708522, 47.109090602386914, 63.243725522201366, 31.57567265944461, 49.63090861016291, 26.50906982441806, 38.295767004749244, 40.10663333940031, 29.74308198099828, 53.50497700081592, 14.467889947467341, 20.91192578212744, 35.848839002274495, 22.877388641525002, 53.86142245395937, 44.75349253779759, 51.798327487641814, 41.22522761440173, 36.36873250984295, 78.10327505611832, 54.1611170658679, 15.615996320746863, 37.14420159911495, 36.036929712406504, 35.995209479095294, 21.613308456517963, 48.47113029497753, 29.017123399686234, 55.009938932206424, 26.21755088998049, 56.1471346500568, 22.22014180344547, 29.2525739180272, 31.599666569965528, 26.363748351644812, 41.0258982556997, 31.455452221453793, 30.550035279678717, 28.566653738546428, 37.28518224413627, 33.81787622892412, 19.66741304794888, 48.69932602864766, 48.28776454978302, 78.09396728666287, 82.4010404966117, 22.911418223821723, 78.90843443590963, 37.665028808055496, 30.35047001368679, 23.74447471241015, 17.338395547772862, 51.627284104708885, 71.32560038301934, 70.45181918941931, 49.377303630848836, 38.548890848499845, 15.720329712553967, 26.21755088998049, 28.026661248580712, 24.703633865170968, 54.82249577272572, 37.28518224413627, 52.43457234042203, 17.522973788894276, 41.888398543665794, 60.04922121908241, 46.76208946113622, 70.59650853854625, 48.47113029497753, 30.098989767587433, 38.02118774533163, 17.38477005043379, 41.923927911317286, 50.85015493542874, 26.711072588763262, 55.54188123738939, 41.31550127875385, 46.428332556437624, 70.65841680802657, 48.24181958140915, 44.353704016052234, 45.96324816436878, 31.93342641027829, 22.740770730402392, 32.216564504472615, 64.94331544338195, 25.1068269667957, 45.49317817482288, 42.083443161888646, 83.86702382698186, 42.55818313849425, 53.559982159488214, 33.210000867529615, 34.5017351494932, 25.19888227489151, 32.543580596818664, 49.58625007277215, 55.09007556956941, 60.60785760261358, 35.064351728552346, 10.685013524654078, 26.939931652595643, 91.42210564842257, 33.48161961666991, 43.21704044895575, 39.14549456015172, 38.217533890647886, 35.27820610513885, 25.533314287731123, 32.004467665645805, 50.73396497381334, 68.78661147922686, 50.22235962484869, 49.40720771419506, 54.17469133394858, 20.261409050319717, 34.69758253145731, 95.90880129593094, 25.77353972543533, 33.594104188700406, 57.98684188897476, 29.74308198099828, 60.06141630166105, 28.48635480649828, 54.86270961299656, 23.18158592696132, 20.798766116401875, 40.31077633565654, 50.33976890662237, 28.405810668530975, 63.10478611169294, 51.19706173682602, 40.16241883628273, 27.641882066000203, 61.23300906621427, 22.184972165859385, 50.3983796796233, 25.19888227489151, 35.085807298432314, 50.016174414053175, 21.43119927846537, 60.37776738868753, 49.720100015244356, 90.25072241024671, 106.02188549851535, 38.27621579026203, 24.64093768499214, 63.16271682737045, 47.09338425626923, 48.48638376762557, 31.383100368053167, 71.63136260383399, 36.47176730572635, 26.911430134941217, 41.74594653018571, 54.55355073621512, 68.32908307952428, 45.28079007849569, 30.994060283280305, 40.032136214500284, 43.83162706211421, 42.99282942660889, 43.182629687460086, 17.659990198717153, 8.941122112401718, 45.11670317075831, 48.60819033808796, 58.65348426059487, 39.2027255870755, 71.83446525475419, 71.76344852080659, 17.38477005043379, 43.52548147988726, 52.39240395091819, 84.24664875461325, 30.748232139298366, 88.39423393961293, 58.12584913008076, 32.91303591228398, 37.022897521708416, 51.0817078509576, 23.381937583677814, 61.62632681137718, 45.41161485853055, 50.22235962484869, 116.26106063674031, 45.7531477097932, 28.593349919071766, 58.3525878272219, 16.42912718450296, 56.91365811108909, 64.02516330142605, 47.375405307435685, 71.66186764175944, 30.822206014644703, 24.32494604586549, 39.99483271762272, 51.670103503418076, 41.36956908425547, 40.89851599155605, 21.9725184489257, 98.59886314845664, 74.04056542361369, 29.74308198099828, 70.07836661183808, 43.576667823372816, 45.476871911797225, 28.48635480649828, 60.668258540427665, 32.31036803681751, 42.48819356915206, 28.21692426339143, 76.5808109786934, 38.87726164502172, 54.8090762865482, 60.81296505884737, 47.81066572566592, 29.148187602682558, 57.974182535425, 84.42722076540586, 46.34849343491575, 45.80172434434105, 32.59000480596934, 38.295767004749244, 58.3525878272219, 62.12319368008026, 57.68232458176074, 75.8173734490224, 71.40726398998197, 38.217533890647886, 45.607106947120045, 25.593601028178913, 37.665028808055496, 41.56716232776899, 16.07922471382004, 14.86208250343784, 48.926409488333995, 42.819521534173916, 41.351549352022616, 39.46862176151079, 29.972398479280276, 51.812559870225414, 45.460559515189956, 58.2896972785619, 48.53210039975724, 43.72983384906252, 31.11619309113288, 29.94700199166568, 54.10674782625762, 39.807744649493664, 56.05547021731851, 64.3668007674161, 62.814303281164484, 50.515362019087135, 38.11949464339053, 37.54550768954131, 55.3430257921785, 59.92706285422964, 76.66620508766464, 20.760916413380734, 25.168228803826167, 48.71450481378197, 50.104649861373076, 40.67914876134771, 65.71499311500742, 30.698817307392854, 25.77353972543533, 47.64016231303186, 68.73357104778948, 46.059868824062306, 34.017921339353954, 47.07765826694747, 68.7123449597009, 42.31271032559974, 31.43136201111693, 44.91897165554234, 35.299523427840455, 38.256671612931974, 72.37990662465891, 64.26451079804356, 48.94151467242513, 25.351488418251396, 20.455134224200787, 43.916273020948125, 32.566791095545724, 69.39882703410636, 59.59594085601724, 53.28440246298923, 29.33061265031755, 30.674066690534183, 52.840363760411336, 70.87467998026892, 14.295032203102984, 17.151552476447197, 12.059015802945348, 41.97717057794078, 14.11968674144331, 33.750906763013504, 30.969584933839823, 37.54550768954131, 57.94886661925031, 35.932570282559205, 22.255281643324963, 47.77971625421159, 42.55818313849425, 16.28021415374214, 47.81066572566592, 35.91165219903394, 31.647574366162825, 25.41224557686599, 19.052604739547647, 62.814303281164484, 78.36333114453386, 32.86709928430508, 44.18600183729437, 38.742408665725414, 62.158529123084136, 36.20322542420999, 35.256894278926254, 38.64577788224578, 24.356732877332142, 37.56546184087498, 46.71457299786424, 55.03666061693318, 25.74365854967445, 25.623672897812916, 52.15279553021306, 37.645130566911604, 39.24082488320104, 39.54424034192949, 135.31999957082124, 24.672307552624122, 56.27781811714589, 15.298189471244998, 21.613308456517963, 17.198459059704234, 30.45042830748428, 73.94220478153889, 38.83876788846971, 49.77945933739216, 20.532046974557407, 29.304616818856395, 50.951589525222964, 49.04707416035155, 66.32345004772485, 49.95710876251531, 42.15412426654448, 38.3348202031543, 54.14752702300988, 29.89616229088037, 45.21523617779896, 17.104456701311786, 20.98697746318211, 48.33367762281854, 30.62452333237271, 32.61317997957869, 51.06726300259586, 31.50361672795692, 76.55232165684534, 29.356583679295074, 49.55645883460157, 45.034424497316195, 74.45221796934996, 43.88243826823135, 16.527533365830248, 68.7441863332483, 55.54188123738939, 60.92125858158354, 45.54203091581075, 62.52242372966032, 59.755610662464136, 28.93817279433851, 43.72983384906252, 43.71283644764345, 55.91110212114217, 40.73410737270218, 33.3687302714156, 33.90693609463788, 16.866809701815622, 32.26349219063734, 32.520331448520196, 22.97928041766652, 60.83704633922186, 40.752415031571566, 31.86221552567518, 45.94711675545636, 33.50413660650161, 29.845229399200676, 42.10111719008858, 22.430006685057958, 40.642471645952234, 30.550035279678717, 37.86333092070562, 28.108380034854918, 27.25118931457014, 26.968400491720537, 32.051727773146276, 40.55061747623467, 33.45906614620007, 44.968493771574444, 29.356583679295074, 38.45174551499929, 25.076078347120692, 48.85083758542903, 51.42698588382012, 46.904379064057636, 24.229253912771497, 34.978428130828334, 33.14171957525534, 46.23647886666591, 27.834991079744107, 19.58669556602245, 49.868359986559994, 40.752415031571566, 37.684915919448876, 37.74447493087796, 33.27813257356623, 28.29804934808077, 27.834991079744107, 73.20017111484377, 31.455452221453793, 18.152361506281704, 54.64783568099033, 45.24802558037033, 27.862470467648627, 38.64577788224578, 22.22014180344547, 41.56716232776899, 54.05233393340843, 50.35443157422832, 35.085807298432314, 90.5869868698315, 51.12499087792484, 32.28693921564989, 52.728742318519565, 44.38716161268025, 33.526657721690974, 54.96982505521667, 33.616540788534664, 67.67454893992112, 23.481423975181023, 44.58736876782, 74.39355021994163, 21.901204463221966, 16.770681005558465, 68.95606185155346, 24.101007033794495, 39.99483271762272, 10.685013524654078, 13.51578700676856, 19.052604739547647, 44.00074794144847, 63.23215496110763, 35.89074000802528, 48.13439940326619, 34.5017351494932, 25.7137125746862, 35.15007114318006, 25.41224557686599, 75.76939588105334, 26.538040802140245, 67.92209946040684, 12.269987677665453, 27.641882066000203, 46.69871340583398, 49.34738061643553, 19.747761138113372, 42.5756498168487, 65.6037363113584, 16.379647304393018, 31.671509916964663, 52.95174014152769, 14.467889947467341, 30.674066690534183, 36.92149399507868, 23.777113143577072, 54.98320374256049, 52.39240395091819, 63.22059262676969, 24.32494604586549, 42.18941807590665, 44.05135426112308, 35.806904786531085, 17.840835911592805, 31.957132726335463, 29.174333687938034, 46.28452400800003, 14.917426646625241, 38.85801980704926, 31.743157602495096, 53.46369996398394, 78.27055351856438, 32.68264174109964, 32.72885510691875, 21.64953965384768, 23.31536402741127, 23.580449387239195, 39.107297094330704, 39.35488893248995, 52.25159499556714, 44.984978533670905, 53.943322497161425, 19.8277448619738, 26.882921524190767, 52.1669271291488, 42.715191151560894, 37.30526590044257, 34.17265130052676, 72.24899070938318, 38.819523215514906, 37.30526590044257, 30.42548305710147, 23.348691877543416, 59.12665430025734, 47.31288915090177, 30.62452333237271, 51.45564784956998, 41.53131141820444, 35.38461788548268, 38.432278259255305, 25.014429475916423, 35.04290151132277, 25.1068269667957, 26.968400491720537, 28.162708678093328, 28.885417652280015, 19.90740490246824, 34.740961884580464, 25.74365854967445, 33.0961219704633, 68.05692852392508, 32.095347902926996, 41.215013579878615, 33.44585620241667, 25.698482533484125, 40.22639748518588, 30.3437619910167, 29.40211251603576, 48.60364053353279, 36.751805495412846, 54.272132472526884, 59.31546163121274, 41.56994146755794, 16.39571386839564, 30.037324491339277, 27.64919307848668, 31.99577724350115, 80.3904823657873, 35.553763464417756, 28.149231645931266, 37.26497703964895, 31.50419341823481, 35.800687657095374, 36.23096559419779, 40.77966540670583, 45.09827810261618, 35.18496850565373, 44.14506607757718, 25.41676012029085, 32.90193460657791, 40.34810511468964, 25.754407990929003, 32.65349684587648, 37.29326397619781, 29.655531028051612, 28.060183783213617, 56.329516846577405, 35.26506272299373, 25.92145888090598, 50.810506820549065, 35.62308249462197, 30.355495220095936, 28.149231645931266, 25.935342968881308, 32.062201111304645, 59.12902948369981, 22.274486905217888, 40.37413063137, 77.74063717233976, 22.58272188878784, 54.62799330880331, 29.655531028051612, 47.9630391232418, 47.81058798727338, 31.312076997153355, 24.014827724200504, 50.019067292160436, 31.413947320467546, 72.80952350476183, 34.37331020842292, 36.26012004622038, 56.28655222330706, 49.69184957395463, 28.111112109101477, 24.462250422531614, 67.18779622317602, 19.52332477345788, 34.88292620573084, 48.924215160365364, 17.479201459214835, 18.391914829240697, 40.77108791024218, 23.326270551973106, 19.673778255902732, 41.35058607106705, 67.2338289066467, 37.99339185846816, 22.339772863255643, 18.7313615001207, 31.717455143068104, 37.17996249750601, 55.10712392492822, 43.20220242437163, 44.467777087718844, 30.22629766158763, 23.51202003950923, 32.348326697765366, 20.53317345252396, 26.551395457044634, 32.923446809076516, 46.75871183362338, 42.48483707982973, 79.84370560225308, 37.76153086759754, 52.86978695523318, 43.28294654769226, 26.59193995481218, 39.937963780329106, 64.41204501308478, 25.360005814307584, 21.251878089462647, 57.543090421959164, 54.86809945634101, 29.81126957057967, 31.20985369587598, 39.79733337813408, 26.61892425490136, 22.870585801041496, 37.60301602422244, 55.68143787074607, 36.627126129954746, 49.080158867030896, 35.761295967594265, 38.14100564581235, 42.681740792011134, 61.792454440625626, 46.51993676798187, 39.665027358720025, 26.780192190189247, 29.048264425526508, 25.684460700758155, 26.873788082820283, 41.98006062756671, 21.626760571372948, 38.8250850596811, 32.26056154578077, 35.05443256641384, 45.21404007901847, 22.56662602418658, 37.49068031219619, 55.238771434923606, 35.56368567642807, 22.806950368089108, 62.14275740533953, 63.09441148243147, 47.23249657287089, 74.39830819665148, 47.752384236871706, 55.25754664685566, 45.05962682851055, 23.15456778426623, 22.404851864966183, 37.34036479813507, 50.226776473901204, 38.434448404600246, 36.32802060148483, 36.49236326806618, 58.63082115216469, 25.160306716676107, 20.764016024668333, 44.585245413172245, 16.958211898120187, 37.49068031219619, 48.69648305974302, 39.850136980116794, 36.885571575818055, 18.045372610156445, 64.04763484329162, 53.933105900580934, 47.144042612400156, 49.186178826140015, 31.784495782567898, 14.819923092753811, 36.67512687820305, 63.16533057590077, 30.889537430240683, 39.585414577769875, 57.242193641443954, 79.6376739692337, 63.12715596007799, 27.020172502961838, 62.29227067126952, 42.146264641114286, 47.01842367003957, 24.872010957139608, 58.51295620186539, 63.61617962520827, 23.326270551973106, 26.980317814184346, 63.92928698068993, 60.118407976964676, 20.044411827973512, 28.8510835082816, 15.175030825644988, 47.408889740558145, 52.37606859519951, 21.9283075619503, 11.659140796753928, 33.93895965040294, 28.034682531748107, 53.97800881544848, 43.40374587843462, 58.465748306982434, 18.532525959008957, 71.6265495319476, 27.545432250902493, 41.41821437390896, 39.60312267177596, 31.515446866198218, 28.81393116825422, 38.97840554595457, 35.11473761655903, 14.63868363604906, 24.770256408448304, 40.295984741803316, 45.206339067023336, 33.084286134031764, 53.92669327540062, 20.568875203348806, 24.014827724200504, 22.274486905217888, 26.320489051593697, 47.42356070658259, 39.61196518760489, 81.59444262000464, 44.98996145202503, 32.610080327144594, 44.27129937537727, 33.980582800755315, 60.23875461572637, 70.59846878960722, 52.46859495944883, 78.0396170155071, 34.33218968960957, 20.044411827973512, 64.29973607468419, 45.962829420304935, 22.46969505155847, 34.85257590272307, 27.099643110848728, 55.6752333838875, 47.02583013426077, 29.739492893350587, 18.92792478150755, 31.58294237735121, 106.60347903370302, 74.24608069453991, 48.959712021398026, 36.201804938305365, 40.269906344449346, 32.24958356634329, 54.81132798441867, 15.373755924409824, 60.18146984732509, 28.83869721798164, 42.04661512390061, 30.10834575066027, 21.7277941506501, 18.86917495259671, 24.314107372589678, 58.23501742554831, 44.24764892030856, 46.000705123839204, 66.50890388480265, 35.3549306394647, 47.781502247494494, 44.88914628805713, 37.74291523050005, 14.922329128070933, 40.63347545360964, 21.26906149860557, 36.52127982919397, 29.10961762440029, 25.64239912869753, 15.200014926482595, 22.35604644640554, 35.830192135329256, 36.56942674533957, 55.64417657229141, 32.11743262572468, 18.8495715703863, 74.17215628029135, 41.77971628825097, 45.32947950094417, 43.88350527454443, 47.34283286820699, 53.72089932044148, 43.96294437989713, 20.22696632238121, 31.918108089827495, 29.450568420721932, 48.028223355279906, 51.06922420644541, 26.415833629321124, 46.63948327439795, 47.533435942902386, 65.0023665468218, 31.56046088791834, 30.92401934943493, 27.16568671833042, 47.59191804884566, 46.68423642730956, 33.990990067752485, 48.82471386795977, 23.310701720296816, 76.55963101198289, 65.17152964925215, 25.174643112262167, 36.56942674533957, 30.739604176806203, 33.980582800755315, 58.151961639459785, 18.632237506757175, 36.47306603716107, 39.25634690504793, 22.51823262341422, 59.634505765481315, 45.22945266746874, 46.43757437356318, 47.7305405439206, 30.854992656131305, 57.9497383046596, 22.4210740647082, 22.176196828294408, 37.07577551261864, 24.915502551324753, 40.00809316026331, 15.34909243693892, 42.95905037296117, 54.28488464328889, 37.910092750246605, 71.80838114225656, 72.92251058570413, 51.64308734132192, 48.33119968779563, 73.3586269061494, 78.77328465354957, 16.39571386839564, 35.22503961207306, 20.371682960503488, 35.374870718774034, 39.1490026808917, 49.803495845535224, 60.25020483114585, 40.70235073048552, 32.880407509689995, 45.69687127821788, 40.3046846401792, 41.35058607106705, 17.775269320012494, 33.719577293982006, 35.54385720580421, 41.54468317152644, 28.389425766429156, 25.81022965638242, 77.8770852625608, 32.934187107492775, 26.3750243857263, 51.535571581583966, 25.260368841627738, 62.968744189607534, 59.56504905800554, 29.954251151768567, 27.25786482209566, 22.011264798057717, 74.5134233135669, 35.494244413339146, 33.073597400688, 42.787987035235005, 35.34494666174328, 62.87569753336384, 23.21716990683657, 35.95778136650761, 41.90504484366282, 30.049162499981264, 35.9773696239017, 60.10118751617701, 61.98731894949631, 85.95304609905105, 28.199988124685852, 20.479509865307, 55.10712392492822, 61.65845876731558, 17.479201459214835, 45.45995927702818, 34.06366795864185, 37.43439748958766, 41.880013669736, 42.303517945136576, 44.11344513476169, 47.977533993846855, 26.306849242041487, 32.424904418354544, 28.83869721798164, 31.1186795726598, 21.911687700196737, 21.878374962480745, 20.0260732308353, 71.41061862333677, 45.98556162156393, 32.424904418354544, 28.25064716953981, 33.35054926365317, 39.60312267177596, 44.66336883752725, 38.434448404600246, 29.70354670451964, 19.102903129099026, 57.11532674856855, 52.59389502234528, 62.73859870846069, 33.980582800755315, 17.002275182879025, 43.33130232052873, 25.088574343778657, 33.719577293982006, 29.048264425526508, 35.84985381411376, 57.69892883785168, 121.52928014976494, 69.21818525488767, 9.52633164593093, 35.91857110363568, 86.53146264095243, 33.24432275848955, 36.006721663616, 55.55088660531241, 53.60477935702504, 33.2974808069638, 40.13049414048867, 30.51907952985591, 18.311016631746035, 29.32926868238012, 20.281363621623235, 13.173007920090884, 56.268120086692804, 79.66346080776299, 9.740105570168307, 19.371618624815728, 25.174643112262167, 69.91919459418848, 14.375199959660398, 19.598710578707777, 22.50205582789942, 84.33541265106984, 35.967586218222955, 40.7109433832148, 42.98344101916359, 49.67087775898778, 18.888793109111475, 15.78669669094014, 52.07749987953608, 32.523083895174665, 13.691034271754651, 30.808905532989222, 44.499135568532466, 51.71687245797305, 31.22121976559113, 29.523070448176902, 32.81575707466591, 37.28384348563813, 29.170809979283987, 29.256254712948035, 42.62441191936754, 21.14835823851078, 65.10814897896704, 42.7062812599367, 43.33936636156727, 70.66651304350111, 46.41507253335505, 51.30632354679019, 68.14775575393054, 48.74639599241834, 40.40881501907025, 41.94673406368973, 45.044153698712584, 22.241765697861695, 36.99030376076022, 24.68269383559108, 49.242636646221385, 16.44151474770215, 29.389996779125834, 63.621587025609486, 44.842526310028276, 47.59191804884566, 57.931863755520105, 40.16539884676289, 67.85465052782352, 119.75589634981387, 81.5818663355842, 89.35479428107466, 40.02560539677045, 86.44060399243394, 69.60908268449663, 37.17996249750601, 38.434448404600246, 35.16491490008424, 41.138538562008186, 71.54026288615533, 30.716466803557402, 46.257312298599764, 67.82931376361599, 48.36714868755732, 75.68638957525307, 78.6297188940287, 70.6081948395471, 55.2137196038092, 60.734712065598586, 46.40757891348901, 57.09717091698475, 50.653244863261804, 26.52436318499623, 40.91676344014157, 47.03322130820968, 47.504159855518346, 51.72356898977847, 36.69431233369398, 58.56602515890024, 52.19713966632026, 20.154162531897995, 23.403857894080076, 43.73217904298424, 70.15452986762584, 28.389425766429156, 48.29523658224982, 55.44497289559744, 34.445136690008155, 54.70393341866671, 31.232603081651853, 68.49477441037435, 25.331577308769166, 40.21767985363428, 46.65440595448938, 44.70237718999928, 55.731095290319544, 72.78125535636399, 70.9428642027412, 35.92838722349434, 19.952463295487775, 37.48131195765245, 68.82501726549687, 18.908351523606587, 54.34858695182537, 78.04400134514702, 48.07163307107836, 22.63099597596073, 46.810774110537814, 26.699676047117165, 74.26916495978902, 35.19500032580364, 20.443636892707737, 40.97659190954364, 53.54016110621412, 107.67371922152061, 58.0033391668685, 46.20460097268272, 77.89467526292485, 59.94605048388218, 23.696193116319144, 26.873788082820283, 68.55994654869065, 36.114155682765215, 31.627854505784093, 33.308093566593826, 71.3096200537425, 82.57782486987664, 58.67789201485148, 28.111112109101477, 50.48172992726794, 28.702213533526006, 50.067612293042316, 33.32932856346112, 36.92368390889546, 41.54468317152644, 51.636379993000034, 36.49236326806618, 63.355848481483655, 42.77983000573595, 53.105091803622024, 37.64036522830507, 33.36116435020209, 50.78318979959997, 35.879334399842364, 24.314107372589678, 30.167383033487617, 33.813785289988196, 31.67267777028037, 61.5689756037537, 22.192601011887934, 44.865835215981804, 35.034300171731545, 40.06058917356908, 53.93952994919199, 28.912859583185654, 29.40211251603576, 60.63245035391793, 27.454287825854244, 43.1536956559903, 68.12254058007522, 47.129281159583684, 77.46259338573715, 23.028839461445905, 65.90635668588176, 30.40231319155567, 29.35357136564799, 59.22231568719636, 38.48008346357075, 66.93661064173673, 26.07358946928894, 32.58834910122768, 38.434448404600246, 37.80801900115947, 25.796299689575132, 36.80918602373873, 73.92674206931095, 94.26816609688004, 22.093882757629643, 54.32312257532422, 27.855499108289372, 34.567898621631265, 34.47586555256142, 72.75768697127626, 43.27487153419396, 54.34858695182537, 32.006847493658306, 98.29402679767934, 20.515314040996806, 15.78669669094014, 29.40211251603576, 59.391035217760425, 50.24749446086385, 41.90504484366282, 16.555372972435954, 54.85548749099642, 31.728649654273525, 47.335477976662034, 20.081068646186434, 53.436596277716, 48.58219439858451, 59.02388678755485, 28.00915645835678, 17.35057109905908, 19.542219221815618, 30.554005861641734, 20.816876360567157, 34.45537635770956, 75.29141677053883, 26.645853603357484, 56.231260480915445, 60.42166791390552, 49.81045931276484, 33.24432275848955, 42.942802543402316, 15.520944499249953, 51.252235172390975, 23.634965053850195, 81.62798032543093, 48.717876942301196, 27.467317647334884, 25.41676012029085, 24.697301589695368, 47.72324775971882, 21.978116537736717, 31.605406903860068, 76.59989561588282, 35.553763464417756, 16.802843248190584, 40.547225581202795, 38.002627662888, 44.40500581358088, 41.737836738793035, 27.32352381634528, 27.983629813372957, 43.13750894662821, 24.521248286435444, 28.87579204131723, 57.452995669987246, 25.796299689575132, 45.70450073290954, 20.0260732308353, 31.50419341823481, 25.374220713922448, 68.19815737039387, 60.25020483114585, 32.051130072771606, 89.20565081264039, 51.857425944527826, 35.13482141691624, 34.771498698947134, 54.048496038240614, 64.11746897617486, 22.093882757629643, 71.53546047299939, 30.39062145725706, 67.18779622317602, 39.726830977482514, 47.62843319577001, 25.949192698330585, 54.57729780955933, 22.806950368089108, 83.77414541639907, 43.41179587710521, 31.895877051986023, 61.992872702376, 73.115100014477, 27.28414372977373, 20.568875203348806, 42.9996882741307, 17.754321048559774, 62.71663174937196, 57.92589674408341, 61.24342423960343, 40.174111764189796, 24.432693792817137, 43.41179587710521, 21.11372825646541, 55.14700168576422, 60.74198748414964, 73.85857446569939, 25.93788451216172, 48.75843572157821, 33.41511569758931, 63.15837080299959, 65.17473053010008, 46.74999573103861, 73.96801492304347, 36.74220402347332, 28.483344046229202, 39.64616673621778, 30.24289744768019, 44.98818515155193, 90.18960266001332, 55.79673386980672, 22.712447861676985, 39.159012481970976, 64.5158426787079, 27.460743667822953, 64.21600125799142, 50.298124173818046, 49.35626706777898, 64.3963193555414, 65.10523977108814, 38.57031120912008, 101.09436822521634, 95.49849768376276, 39.03011188175203, 34.41322398049752, 31.782309679200022, 29.023800581640742, 32.81161726876392, 33.51579989010586, 64.92446368920378, 45.87854612153218, 56.780319733573975, 57.022139334553806, 32.96537041219809, 28.778667546240033, 27.366634183802937, 38.857555945669574, 60.24007912446466, 107.1675225450981, 54.44997179400058, 55.6053348410568, 43.35822480192832, 62.51103085768008, 35.40979422263239, 41.010040862224415, 48.44685479566951, 30.960133565256893, 65.56587528893391, 64.00166899856718, 86.94416855102428, 66.61706772029764, 54.80396315284386, 60.61231170242685, 35.92428553245836, 36.51479742442424, 40.13327610576763, 48.688096805152625, 31.66583710054961, 51.338911423315956, 59.45513772113095, 34.033588756993936, 44.826478866665916, 65.21127350877232, 23.549469173623642, 23.710498275262122, 23.255337105254046, 26.373019095697714, 25.25634109194286, 33.13294255166756, 47.32473095583051, 74.46277705531578, 61.39007044640327, 60.34817627640167, 38.312196952080164, 48.08183569697074, 42.932535544212584, 20.0942415831872, 45.74620504776051, 24.34362313762745, 67.30588151779503, 59.15352609269601, 25.160178965601048, 35.02287569257609, 40.85382887756255, 57.54832028731026, 94.79982064196142, 51.047982557831205, 57.6835836474251, 19.308748089560233, 80.67929726531477, 51.32960747305212, 32.50434315684002, 55.909253048763354, 34.6893765849579, 26.40055405282657, 31.693722806400334, 13.513539158744889, 43.57497853367091, 46.06769086885119, 84.96115048915092, 37.42450372267921, 50.96833573880393, 20.086164903327905, 103.0211652908563, 16.410831236518284, 22.446738544906594, 27.828098534480237, 30.272124977186785, 43.45020504328451, 20.73580120339154, 44.984634511877765, 34.445838692978064, 39.68849659971111, 41.054836737895, 78.70117181141732, 23.95501589401334, 18.47532088214722, 37.548361552432, 67.07430889383019, 33.29482945369769, 27.328313924524252, 64.72968667968125, 28.987725309024004, 18.821080860295762, 66.69567470444278, 37.6951633403966, 25.16980698045622, 43.77058902791352, 59.81370502748922, 28.898712385437708, 74.69565242369775, 62.9328168658734, 33.81132485934396, 30.415239255177966, 31.089662245214292, 43.9362445878369, 60.702722673660176, 20.90369121933392, 22.45396808649116, 43.49979515612109, 34.6153500779731, 28.018569987983202, 29.68452259757665, 47.09332008986087, 15.312357729420055, 56.33590170575546, 40.57307849316603, 39.364764651222615, 23.487504832419024, 71.9083071758268, 18.48412914346114, 32.66696388038603, 82.70742753594315, 49.15079347199061, 21.52784384157404, 34.33156775360735, 27.61587613802483, 26.598477426493265, 39.96957287112414, 73.90901948512357, 40.63214145668611, 54.11569875293905, 39.87343431021925, 51.4520439509283, 81.05950116330963, 36.76616498786476, 25.85058194201781, 34.758644327522696, 11.127102723385102, 37.82235413025207, 40.18304963435844, 36.10667915783385, 27.91195793078984, 25.561550303043738, 17.540279391880244, 53.52265764681482, 45.39605169632795, 16.355747573340373, 35.052613758457824, 8.696618638851929, 20.708329537849345, 20.295726436599473, 62.26653266503819, 16.844854485520695, 38.20553110018641, 67.78069421991529, 25.150521122424674, 31.888285870724825, 78.27397918244337, 36.495043587322876, 25.005533468857777, 56.38953024436233, 30.772627920290155, 64.26545544710571, 44.433056490598084, 36.25945111653449, 38.024968118651095, 37.36240921361377, 46.37831620735709, 40.8557947901934, 27.284024075856298, 33.47509327373703, 13.605175402527216, 30.080250433768043, 32.767555232457624, 50.33929640174737, 44.433056490598084, 15.636731426430226, 59.42839052309078, 13.525805079613455, 32.85807379722829, 40.74216877006606, 34.18408853985294, 29.04042830748428, 26.348532551734923, 26.935992520177717, 66.1914126462588, 54.45435393051813, 27.857041585833166, 40.67933537276845, 36.909632969476576, 37.252971874957765, 44.55683372900127, 52.32679357712423, 68.83806150179737, 57.05142958364806, 19.522805367843816, 65.39245854033629, 24.400045815124628, 19.070796035203454, 27.118028897167694, 56.65410318319879, 52.04145049879957, 24.183601028178913, 52.716394857551656, 15.32311487269698, 31.084487509429103, 77.9685596461333, 37.96597548960332, 49.9548169698772, 54.174494702218055, 28.92376084244079, 25.43799873760939, 27.71493848608956, 44.11033189167912, 48.9037056036659, 52.127073587890955, 44.69810682080926, 75.50130948361891, 41.334201294625, 48.56352467725473, 19.10925626149106, 10.680575924382092, 77.15088210852343, 62.471622889725566, 41.494348067707634, 25.618357660560836, 31.2957675531082, 27.310616400224706, 70.77287321500185, 25.884907347146584, 22.826361739454747, 27.56037815788592, 47.613630991440424, 16.941880480875486, 43.22911826107415, 50.77574610666447, 63.07781686978692, 17.479533180837638, 26.188759167500493, 39.35053617438938, 49.040414096753615, 22.298132291559636, 58.8179135996675, 21.193242078126985, 42.17802716664628, 55.51369373909947, 36.09114084088184, 39.179433710575864, 20.802321089544247, 28.646856385160913, 32.28873243935731, 10.19656939073981, 63.00851114134052, 82.93892584164384, 29.924737242440916, 34.153578420177354, 55.398917727302134, 23.79908187212971, 17.446720717805622, 43.66101225090262, 51.19754591354523, 43.22173788744853, 55.112357576547865, 52.3709080565225, 20.716186976113214, 85.16136276497501, 36.54550003118729, 80.24495312883555, 64.91345004772485, 53.07910083991138, 26.253416225600716, 56.63303940104411, 46.63214283423754, 15.398151125682237, 51.25818672796146, 58.090200843995056, 51.194431902171736, 39.87343431021925, 80.1461313865352, 20.130676151931212, 36.00447593903455, 54.59884724189785, 50.4988768116969, 19.783924285547275, 66.42251215070799, 19.62272013869136, 28.35581145436209, 42.59450146811058, 61.09551858095326, 54.18624375564431, 24.83360100241265, 43.69756887102586, 38.748286283672954, 23.992134944934065, 35.63089035212196, 25.68133110336672, 22.97850293106724, 30.876939215649884, 37.69941351900289, 47.34830566480778, 73.31860319146273, 58.84629926382564, 108.66954301118226, 19.126297056023006, 43.10148038194605, 26.201094569330543, 27.653787572570383, 50.74593764379146, 60.2519570734809, 60.90922752914103, 42.0775271254153, 21.663177458264936, 16.673640936396644, 40.793139466372104, 41.278090030476996, 35.57237514613045, 36.41376214985798, 44.26207114941751, 63.56331891603438, 15.478105863428464, 59.86019982784639, 35.33728903963654, 97.37675214292354, 70.69771722027693, 36.239555351396916, 41.4558152792074, 58.507548625791856, 38.06283059522094, 69.73035126796779, 24.106538903108408, 101.5954793563782, 53.92416335169669, 51.55563721470102, 83.41896705467923, 52.384582181539, 29.178689658105274, 17.427969025097624, 39.0198628398971, 36.41376214985798, 47.14070619173817, 27.810727264929003, 31.405973379591686, 64.86205451597412, 33.52059128953989, 45.58372636256379, 39.003461992852195, 29.483247326994043, 23.321427423220975, 39.4621648435417, 24.250411301017486, 39.05470172786357, 68.57352170712855, 31.878200641136342, 40.06748192363264, 40.37556694667573, 35.118875582569416, 53.978749762446135, 53.146549018877494, 44.52099393597613, 88.82735205713867, 52.319184672521644, 31.216054424936257, 78.04781988140412, 72.7658681601374, 51.76067218438744, 39.512799014673085, 59.27703072008112, 95.59374285082325, 32.47716880994642, 48.086820170653226, 41.0334236344989, 71.03583890282036, 53.605883598766866, 37.0460545865019, 22.04567131200823, 19.6724466522756, 66.78132726458814, 68.3880889122772, 50.74907956010592, 37.710028229512, 67.7795174052574, 53.18847060656825, 29.635699147948277, 43.446526655101685, 41.594366512123, 18.302100173970395, 30.205637286302537, 75.8951572168099, 49.220461633107455, 22.115495102674956, 43.838025580370335, 43.02915586982269, 52.71034852451603, 67.56368783920774, 38.024968118651095, 20.078052004561066, 55.26518948489766, 15.704939917742541, 24.4099825025359, 27.118028897167694, 40.328027332283234, 41.73244991202474, 23.83647861358988, 65.9679227214394, 68.48550367503013, 59.86019982784639, 18.127157789085356, 48.98350146221693, 31.446872622260305, 36.56740930265317, 87.12160940541692, 71.78022129962433, 57.81164944491163, 48.14314456036817, 40.11932502551323, 46.53797845187636, 28.837351240525145, 50.99334176657944, 41.6039704824232, 34.599128188565835, 29.026565055660292, 16.360748403066676, 88.352730473352, 25.996942268623528, 39.5997487089225, 67.99115440051455, 83.29731649384267, 34.118335427092276, 77.62006925712656, 35.294180414087954, 38.104858845925136, 83.31062629394461, 46.62016622768419, 31.59216960666841, 59.507253271329816, 42.4461101682473, 43.76877094540105, 28.697404287570414, 41.92344537426859, 33.5708103985197, 33.92265226167943, 39.18760578448212, 29.414933095715227, 71.88403921331309, 50.261683463473226, 46.47966530485967, 35.47989691582828, 31.213488315852537, 27.425488468673002, 23.737788919653173, 22.70886137807907, 25.485587504420156, 30.391427632538466, 23.466821895565843, 29.77651612727349, 62.46907724956654, 55.50795600082795, 32.647281872793286, 46.73123201443199, 38.853434780886424, 16.239874678151363, 49.05666254658006, 25.700207483112738, 26.77681569701817, 39.787109724725084, 64.46535468368997, 25.67819639669836, 23.80927696763582, 54.79670267257687, 36.763993795306135, 39.95956908425547, 64.05997228032558, 33.263466609781254, 24.237053598783195, 88.7140668896613, 33.90371459084473, 49.42242724934168, 59.40429894960213, 12.598007913188223, 45.067934854014325, 55.54091077668798, 56.048522184579305, 40.575058247301996, 34.45281807085221, 67.65878419377583, 21.68566843465692, 84.74083650067722, 96.09987088230316, 28.815011068858364, 35.62864859006092, 29.8817047484962, 39.96957287112414, 31.77725027775369, 47.78912971782525, 25.034589643148323, 38.68215457503927, 37.90057510218227, 28.658084499986114, 49.86862210444312, 11.791513785406217, 35.53180562795446, 30.670587604232075, 28.204852673240364, 74.11047893980472, 28.915413827547205, 26.021793949328906, 12.181334266779311, 33.96753818354061, 74.71899857288734, 25.30107258876326, 33.350212264908734, 34.83698531737099, 41.85478973907645, 37.222864002678534, 75.34150150099835, 63.10803932965832, 23.02779831383581, 51.55100527896899, 56.630239403302646, 43.919897618627, 47.952337755889445, 40.335946530185716, 4.397228658370579, 46.38691454167596, 19.552004691342013, 25.7190441756233, 33.472684622999914, 37.145354503700986, 16.761913640465863, 42.66757504538867, 25.339367692441943, 15.557630339593308, 35.49118880226347, 20.475307678262755, 36.31683753744451, 31.105160230482376, 23.873787534611328, 38.92749781082476, 32.211531264398246, 50.985531092461, 52.78431273587701, 32.98241327414974, 23.822881790598736, 34.00766208184218, 56.13504176339758, 8.957623375904936, 54.42073261469404, 30.746493513120825, 47.68224520920003, 43.958045654248046, 42.98645152792779, 28.934901388642977, 49.8318498795031, 30.202983293335674, 41.467372693564386, 88.35810491162351, 14.730931015702632, 59.549987767887345, 41.353529889178226, 38.663545280538806, 43.39320660227171, 60.29020486298331, 44.90296309292713, 13.108631325991112, 62.42073139584183, 21.764122182273933, 17.955016904669066, 18.131686961083986, 39.00755039892107, 55.204685146897454, 23.74461343728821, 34.2707878535189, 16.335646828483107, 47.72068500910379, 37.16473859490966, 62.727989138469425, 55.28533491897383, 20.495160584573682, 36.944339804789614, 39.973576914359086, 85.89943448228905, 15.161049608706902, 21.459846195483646, 39.37898789395975, 49.04365368870686, 60.19785098997187, 14.93659499068042, 32.71852188016045, 34.10892339454463, 40.97885325394573, 31.635374566538257, 29.573192261206867, 28.387031138448755, 20.59809559400383, 29.879021292092773, 32.73814025222075, 64.99048611180328, 29.062590926823134, 36.726939807635965, 57.15727638286647, 88.61314985965402, 58.34015482554935, 42.538208858254315, 35.58363709723455, 54.89246973645926, 27.929264618975296, 47.1525485905133, 52.943943251000654, 37.14318843346299, 23.394227070434958, 82.88540968501482, 28.081742791735298, 52.97249831323003, 65.88843818898764, 17.64726340929678, 62.05306857735244, 43.09221567436, 26.849027753011367, 50.753785705432556, 25.24353086841301, 52.27655653132374, 51.076057931050265, 47.35504690526457, 36.20417749665098, 34.62923765979091, 35.86624042420657, 39.09358824239068, 45.18111609587126, 33.66626892216107, 89.968114841549, 53.95367517537639, 33.41750912845684, 33.323722988675684, 15.478105863428464, 21.52784384157404, 50.57612065140569, 21.67817412173439, 19.54366181809766, 29.33000596284936, 51.9142950737732, 42.5213154333081, 32.13416073905165, 32.4450182130374, 34.749427988339825, 17.684299352482668, 53.693411885364974, 54.356357453437745, 26.19799879461901, 50.9089171136348, 33.71865733206302, 55.13400980485831, 45.466301473693726, 24.24373339722361, 32.487043975044074, 17.973233302295036, 16.143549835442325, 66.97955537228624, 56.74389091560628, 18.315476978101973, 51.213100487782484, 36.86621579026203, 42.38395146232822, 75.182410499973, 41.30131843040629, 85.09994824657679, 37.495024189464, 15.578739409088325, 29.59494624987681, 56.057044524789404, 38.562005899588925, 63.716839772786415, 30.66009247328603, 29.164893743760413, 47.968960227562214, 19.97233960349438, 33.723419298730235, 33.894260882488716, 36.74437652857486, 19.882479028217563, 17.09112024279171, 24.27040665135407, 55.018431518682355, 62.488151539435314, 43.89262418246077, 17.02870315070676, 14.65230687858118, 68.99007212565651, 40.86360992649538, 50.41046214281702, 57.51651865805497, 21.33459674028003, 32.59315360599035, 16.63425609604267, 36.80534755352368, 47.23702997124994, 39.58964979037759, 25.3106454842036, 44.188075652644116, 18.75593545480468, 22.008836339933907, 51.402484576545156, 30.417874292359336, 87.57311257647852, 36.3609235910278, 45.11217836591184, 60.49156527652474, 18.054897111316194, 55.04157387075421, 48.43367974720313, 29.081977433733563, 42.15529001070061, 29.12346582396781, 25.46657325259521, 20.4713475844707, 22.016214399598198, 22.388874163478818, 83.51198461821566, 51.84982716826271, 34.46445344623764, 67.76663752761146, 38.2473996210584, 48.159700203163894, 52.10721371154632, 50.897960902383254, 28.304638055413236, 61.073399838814915, 26.56812624471239, 46.83010834789073, 79.41898507151737, 51.22710055032856, 30.253519193146918, 33.101442658159726, 31.66836572700886, 38.915155611013326, 24.64406235211526, 34.31755720308033, 33.0067389936094, 35.904200347940794, 26.803924724654987, 26.049661730736194, 26.545448824139964, 63.699138202747406, 36.06236350888093, 30.85920351840436, 41.123210141238886, 28.850312328766577, 41.9355573891188, 60.70045207005393, 67.89658868573876, 49.221378315413226, 22.429295086110432, 76.6939808637514, 16.456479501853483, 63.92416693334023, 35.14006535908165, 55.53796299142212, 58.46585017784701, 53.4226856096517, 29.3782825364091, 45.960674949702494, 30.82627352683537, 41.00582974786612, 54.495864422640565, 21.44113634026623, 30.75664994957886, 24.332644434193988, 41.74539581118466, 55.22985783002771, 63.16560945701701, 41.427385537961264, 26.53689169798738, 28.280214586153786, 31.208237943925376, 36.520731285328125, 54.59607292956481, 41.23211027920749, 25.575028687902314, 53.293018692449515, 42.941359080645334, 68.21692062039124, 64.20217851092029, 32.738592433264834, 46.85245643250354, 71.50528666607458, 43.152115336031976, 24.67579392877402, 46.11376018032839, 44.634125040717954, 39.416030129482586, 42.653452742763456, 36.753850504559026, 22.373328997319557, 47.846605323376146, 84.876416902213, 57.50087911765468, 17.515900537181494, 48.971020741636934, 52.41454725353186, 33.25072137237164, 31.15759402315602, 33.75491195248348, 39.92504050824651, 40.4136627200161, 31.50680267904789, 52.01690541420775, 33.08091449444391, 18.014791619888943, 34.10989826440364, 89.4408595074874, 28.01088174624296, 58.55156576229332, 37.11281245770717, 33.77825902629675, 62.568778361856246, 44.12207519040131, 25.6060757294653, 29.915571824831943, 33.95444138391408, 37.140046269111046, 29.889124704558625, 37.808428476113576, 26.464189763367436, 61.28663868957982, 25.41925790753141, 21.71623597327292, 42.80288954453036, 49.857366262744584, 41.828539937676865, 72.60367041929288, 28.50728094383032, 37.161219856472314, 73.82561890325134, 25.681285158925473, 37.97143707509792, 48.934491567161196, 51.54253491521732, 51.0855690727419, 30.81164051867987, 26.901632161863915, 25.2355191714923, 61.76631700023117, 33.60111095819937, 43.926262057539034, 51.881514273171994, 66.27141419473864, 45.938763249120676, 28.542963640405492, 56.1411028848406, 38.133716891635224, 42.3291670790445, 49.794582062083926, 49.00523757551745, 48.225911970151415, 39.60627034403851, 14.413510023997167, 38.66289704230749, 55.940127349235205, 24.276524481479576, 84.13314301577043, 38.00098581599924, 32.33683515790228, 59.80931110499633, 58.1562245395224, 80.54590152743452, 37.11886033773013, 55.79840907254471, 32.0813488730156, 50.714780647160254, 49.987159693131225, 27.86101681748581, 24.393299405639127, 72.59448453407887, 65.19818178284548, 36.78745959358108, 34.55896627291231, 38.174926222856136, 67.01266072192732, 25.388000865208785, 43.90077571925264, 11.994237926419556, 43.38765180663262, 37.95368969644359, 67.48576726015206, 57.35334623317465, 26.352594073671224, 22.961464511835146, 45.05837681661979, 27.550512306322105, 70.4006369168434, 55.6723491454789, 37.61187227539252, 47.579489365051415, 25.65476362775191, 42.16738674553298, 30.818957914381386, 24.685017827327673, 47.36989105818279, 41.58400025072481, 27.962362435387075, 59.215830935155836, 36.85154663241011, 48.657379345932696, 33.32855092159787, 94.93759086819202, 29.86265351731975, 27.488803237122237, 52.56765787832419, 28.151920320452774, 27.26545923868913, 61.99127501259224, 28.28820585894963, 33.878103773451876, 48.88194301603221, 39.60061018040398, 72.97013827876954, 81.50092545884172, 50.26571547821392, 57.164499059905644, 26.897412344368735, 47.183054726708384, 71.99640671419354, 45.860773397440774, 42.28944477645283, 18.19263897387192, 21.30755797999453, 28.4596324083248, 69.79310150223269, 38.67740458589373, 23.217886919644343, 38.66289704230749, 40.37201768515638, 44.6967777086191, 27.583381521038646, 22.51553907287091, 48.54009436971786, 50.127737685866734, 22.317217341727456, 28.959809333527176, 38.50870302891577, 68.99667350102554, 42.03964205479641, 71.11555236463515, 78.82500840541073, 56.863357387335014, 31.219080737894693, 52.61646877624298, 31.763472750809626, 33.43987659267245, 29.018301722844132, 59.8949417360748, 40.07926616840341, 62.07923806196271, 43.839523265574584, 54.72669128411203, 64.21084328139557, 44.896647843473836, 16.734855378303376, 59.008464950830046, 61.66345853608216, 77.54948598377801, 38.68610980620281, 57.800573704574546, 47.40764651892558, 13.017746559214828, 84.22805502747532, 17.26409222752328, 47.42414382488065, 60.2565458214801, 30.18612195352681, 48.632098038090575, 15.87706278156632, 39.78714131894606, 30.89941646105203, 60.41164238115405, 24.97774541972606, 47.5512831152073, 88.75241878161526, 27.157256313933733, 9.915948955881108, 82.77665718207336, 32.95126305364366, 46.222790897774075, 43.23250316348921, 16.596260523931882, 26.736869634152118, 38.54366750144522, 29.02220442422433, 45.06831515844397, 30.480168869082988, 46.23488460060192, 68.13045203221394, 33.91130563269084, 63.450483048823784, 31.474567910851885, 24.05069014852559, 52.58463812771601, 27.186441726916957, 36.36963644753991, 52.85352645109644, 21.54736141212732, 27.42283851521599, 101.41921600121587, 72.46575851239734, 39.90256322750021, 34.67273225449145, 64.34417963436906, 36.523814382003756, 36.0218094006331, 45.98987402338468, 38.39777333127527, 64.47205782027187, 35.493558450818995, 22.510466756478177, 47.289564031122865, 33.02298065874418, 83.26862411499087, 21.032345066054493, 49.296234873341156, 58.89505920031494, 54.64305684222195, 48.00986597726364, 26.408440277225075, 47.95862256046385, 48.30465572517805, 67.9244305102621, 16.329588593614496, 33.8382020835057, 47.71558503626999, 28.44372363554229, 40.7810073284865, 32.85196726480455, 47.83258441001493, 39.19622459583562, 28.909009924269366, 44.12207519040131, 67.90641764713381, 55.419356470102066, 69.63838928245022, 42.170049028200786, 46.48821042255936, 26.412740655256766, 47.341557782634666, 22.742115560000837, 34.87656511669118, 21.977365376253363, 12.65244019977594, 54.06422993156382, 36.27059507938869, 9.368997872848192, 45.23684093702044, 27.738879901631428, 56.20065928625977, 44.66420716737039, 35.97809598990403, 66.26133973950334, 34.142888403356395, 47.21857583944325, 36.56686083484939, 30.294415524088357, 31.578328417207747, 63.920687443858455, 77.82825651023023, 34.123102060137384, 14.494033002268871, 26.82571753109243, 43.12356071434802, 54.67979211745143, 25.091149596009366, 47.25645427520538, 48.5516011894541, 30.77130980600398, 36.44988933119535, 27.069529759955543, 50.7345966370676, 37.9862143805345, 47.11426758722189, 46.12588297438255, 44.55129254272833, 40.76175171824812, 36.97938014625541, 27.344265513684714, 22.571139591689622, 28.411901648147314, 66.8331209988354, 55.41331855073365, 22.034375106764664, 35.210403262050036, 63.93287994377301, 60.691268806314135, 35.092016540955065, 31.03786692127699, 15.052550068185601, 44.966395191708145, 37.793580153109104, 47.452426943727666, 30.88845847184061, 21.377131896373893, 41.7400313933886, 32.85196726480455, 15.993280849816328, 34.21862117608716, 24.945890477915686, 54.17761840096324, 62.967886110029, 34.67923451418204, 23.222799209358318, 32.63861339597201, 28.34812428759637, 44.483403672354115, 40.49127251744951, 52.16263725416208, 56.03373953477532, 40.41088238923685, 45.382545056381346, 47.17594449384996, 32.13752938853696, 64.58418132827411, 43.214360517517385, 46.26149902725393, 14.827119199808932, 31.80604452565373, 19.683485202091777, 58.91019056326848, 41.97562108984332, 51.51437370296179, 80.20747419457663, 32.978596298718045, 61.40646250867622, 33.281193510088514, 47.31083936091017, 17.297466831093907, 57.700201974847666, 29.89667410066478, 47.3486427183274, 32.64553098726172, 34.48726821849357, 27.509404658218752, 41.14502315212292, 35.20722097838198, 51.70474733977538, 31.936947900816072, 71.2389463647312, 32.18659217177046, 62.584792583358556, 54.83461342441339, 59.78322962594214, 50.170063845937584, 62.64524646806927, 24.35133037838018, 41.71047561880267, 24.453771927168766, 24.518760920454653, 19.122146848154237, 63.06682115338457, 39.674193531030134, 56.12918391176728, 26.447043254237357, 45.591636055209584, 39.78996612802723, 52.9632127138238, 34.57523663192809, 32.76610752702745, 49.135027465305306, 44.714307916777, 40.213365246766806, 35.87796728703418, 23.74133877898546, 57.36112427007388, 14.21832144867937, 33.523949117732776, 28.147893424817173, 51.73281579202449, 36.76302134399099, 76.59397703915194, 50.47637221077947, 17.548735370052604, 27.517626482144912, 66.58629572730686, 56.81039366744797, 42.30268465515738, 55.52591265828238, 48.16561573424582, 59.119783576534054, 13.700438797719205, 39.92224172475596, 82.45151565262228, 31.749282859288726, 34.99571636336059, 42.27884062185101, 51.609649682959, 64.74596520132607, 35.10163168902186, 26.63919307848668, 42.80551154945333, 54.968711317305655, 52.94424365522153, 59.5984406283782, 44.79932222133767, 37.956653892686, 21.064917049922638, 35.686283219419416, 36.26440254174628, 30.686842878687052, 47.62176687994586, 37.03402787811889, 14.429680702641193, 41.093217719599636, 61.91218648391941, 23.960707387211073, 43.92116894765954, 34.72137498400278, 39.14465848992289, 40.56874686551284, 48.98013879781878, 49.64178504051158, 32.23209491801291, 43.33598849228102, 44.208264053863346, 43.6578104824433, 70.40853011633969, 20.945250041866966, 86.1647962482008, 29.631026179840138, 72.85429487456457, 32.260052999736864, 58.65425683138278, 67.89003208180627, 40.68742948439892, 23.31086743971091, 52.24389400459151, 53.956815607877296, 47.84426428388785, 31.81314782490323, 45.355419892632824, 32.20060471841991, 21.483685678058887, 29.619585918233707, 58.97824940911725, 41.4868556101482, 42.94918366051781, 61.66526805506266, 16.533506674845576, 33.456704513958435, 47.22567954318297, 38.17197933910779, 26.36549631784978, 42.59300745649433, 34.221916637919804, 51.56419736700075, 14.998444873878002, 33.21344529191313, 41.117762939572636, 20.3358250549135, 40.006443072688505, 15.478083546709415, 62.23867584460658, 34.05375714593951, 25.8090930423276, 32.09890721802314, 51.15984160356977, 34.05704980185129, 30.99421208795099, 13.258297385387824, 47.36517459999564, 50.5647962752772, 50.27904691826589, 38.23663816040576, 23.587504832419025, 52.1604929421574, 50.01396987659382, 38.2894532536577, 41.898134528635985, 57.87573387828563, 23.611601921273017, 71.12806354076992, 50.798390491166124, 18.03393711838277, 22.636658825452695, 33.09453899845934, 63.27127343530485, 30.002269722045423, 37.03402787811889, 35.18164100802033, 36.58837353930838, 50.38777763207802, 53.4602722104366, 59.076409025797766, 18.80746087745719, 27.426951154966318, 76.68529011066057, 11.964740241900712, 61.8059738517723, 42.83690227488162, 65.92463603961515, 35.24553683413717, 31.208237943925376, 45.911972532407795, 69.99673346391124, 17.23739023052398, 16.783043805780054, 58.24048480165765, 52.4698977523948, 26.791890195932034, 66.46253516167032, 36.628268589585154, 74.73216595452023, 45.9046619284586, 51.93098018493698, 20.901529349741455, 33.15574875703, 41.76150039125671, 30.708902578056467, 27.738879901631428, 24.578914381024056, 52.149794834393376, 19.607647130879734, 28.991012850053533, 17.57497973237116, 19.51977297348514, 29.68434680864797, 31.47815814156227, 75.52630866404488, 24.458427606748437, 36.242693196672214, 39.648736252746595, 76.12845843933246, 53.397612953779124, 25.6414922439705, 31.40282046428881, 22.04471590343633, 45.522863809281674, 39.62609901373286, 17.344016598549572, 58.764355664081386, 30.431975994174813, 15.811302670547079, 34.99571636336059, 76.66501061226774, 38.79041686080491, 64.2749539039535, 71.21553432696946, 44.53873333136534, 21.446471365797557, 73.21190770610781, 71.95934249513188, 40.76450763905575, 70.31055628917797, 43.63727701778211, 55.89226206479902, 34.571994497527726, 85.60928896800415, 40.118420246850015, 17.489563273001863, 44.023020220110524, 63.403092443531904, 55.69037543136189, 60.55162449611024, 29.297394023693037, 28.862066207347013, 45.313464072626644, 30.990564306930846, 51.881514273171994, 23.35962686620614, 31.363290701003415, 40.014853263733336, 111.72624935940195, 65.34310095196034, 17.14347283692654, 45.89979223940626, 62.403054305016106, 42.755756477329555, 62.42089892561224, 35.953095079432515, 40.377567526532786, 41.68897983008831, 21.925385667144095, 44.38264292245136, 36.78745959358108, 43.09498681458183, 88.84111474961617, 38.25131082459271, 66.14873290430074, 46.71859961180864, 17.042272438008528, 34.66624754444359, 16.449443439008274, 57.52995137610954, 19.619355235559137, 87.22694991084887, 45.9071036325963, 47.61003148933879, 27.824433702553907, 28.272197815584047, 50.4520299094453, 32.448169303474145, 20.154909941717104, 30.406011685961897, 53.25111287721491, 22.312098263300527, 43.07158969559972, 51.25795719566621, 33.07750253901872, 38.17785584227155, 47.05490331052174, 56.1927183014774, 9.940080210190075, 28.554847678145055, 53.22805890134487, 25.082079972960052, 26.22754320600653, 27.24469184898189, 37.02796583620675, 8.589494881207266, 55.902233576721734, 45.06335345773506, 35.8434677772512, 23.73653754451493, 23.998635961782167, 41.50576821594984, 33.80824416199749, 38.16903222297121, 32.85881896120062, 27.488803237122237, 24.41657431913524, 52.290885284544245, 56.92410972855622, 40.502354118866236, 44.3548950430425, 21.312927746766835, 32.33683515790228, 21.63725598222501, 26.57957484070847, 28.622087047898322, 65.26982299398135, 41.863344225183745, 33.07750253901872, 28.704933153298207, 23.149143920122217, 43.47016211461635, 49.70476540320314, 24.210890238303307, 25.813471010667296, 34.02728919533309, 35.19763534563148, 23.242409249200765, 31.671039381171052, 37.130970600260035, 31.012402243768573, 46.73056276638355, 84.55811181191186, 41.60016853665518, 54.45080445868587, 47.5559944156976, 53.04323648588572, 46.20343146868558, 20.604067004606684, 50.5382830019609, 45.70437316082519, 31.19378883207087, 61.75910104362813, 40.94015995285786, 40.64330646433341, 40.731487807186134, 44.61657728206594, 39.38469421708704, 31.17208131492095, 30.17489787868849, 34.50030527028789, 67.64875699557076, 27.69394160809317, 48.98013879781878, 46.94069213740676, 12.680230501696402, 28.30019998887582, 53.483224570756235, 24.388630871733366, 23.227710423144828, 49.025756022737454, 38.35097697593759, 61.96252264912388, 40.535564358852966, 49.30529179559833, 31.535441233546766, 46.6395339350619])
print ('n =', len(sampleMAL05))
print (' ')
