
#include <petsc.h>
#include <petscdm.h>
#include <petscdmda.h>
#include <fvda_impl.h>
#include <fvda.h>
#include <fvda_utils.h>

PetscBool operator_fvspace = PETSC_TRUE;
PetscBool view = PETSC_FALSE;

PetscErrorCode bcset_west(FVDA fv,
                          DACellFace face,
                          PetscInt nfaces,
                          const PetscReal coor[],
                          const PetscReal normal[],
                          const PetscInt cell[],
                          PetscReal time,
                          FVFluxType flux[],
                          PetscReal bcvalue[],
                          void *ctx)
{
  PetscInt f;
  
  for (f=0; f<nfaces; f++) {
    flux[f] = FVFLUX_DIRICHLET_CONSTRAINT;
    bcvalue[f] = 0.3;
  }
  for (f=0; f<nfaces; f++) {
    PetscReal r2=0;
    //r2 += coor[3*f+0]*coor[3*f+0];
    r2 += coor[3*f+1]*coor[3*f+1];
    r2 += coor[3*f+2]*coor[3*f+2];
    if (r2 <= 0.4*0.4) {
      bcvalue[f] = 1.3;
    }
    
    //bcvalue[f] = 0.3;
    //if (fabs(coor[3*f+1]) < 0.4) {
    //  bcvalue[f] = 1.3;
    //}
    
  }
  PetscFunctionReturn(0);
}

PetscErrorCode bcset_default(FVDA fv,
                             DACellFace face,
                             PetscInt nfaces,
                             const PetscReal coor[],
                             const PetscReal normal[],
                             const PetscInt cell[],
                             PetscReal time,
                             FVFluxType flux[],
                             PetscReal bcvalue[],
                             void *ctx)
{
  PetscInt f;
  
  for (f=0; f<nfaces; f++) {
    flux[f] = FVFLUX_DIRICHLET_CONSTRAINT;
    bcvalue[f] = 0.3;
  }
  PetscFunctionReturn(0);
}

/* autogenerated - see src/finite_volume/python/steady_diffusion_ex1.py */
PetscErrorCode mms_evaluate_Q_ex1(const PetscReal x[],PetscReal Q[])
{
  Q[0] = sin(2.1*M_PI*x[0])*sin(0.4*M_PI*x[2])*cos(1.1*M_PI*x[1]);
  PetscFunctionReturn(0);
}

PetscErrorCode mms_evaluate_gradQ_ex1(const PetscReal x[],PetscReal gradQ[])
{
  gradQ[0] = 2.1*M_PI*sin(0.4*M_PI*x[2])*cos(2.1*M_PI*x[0])*cos(1.1*M_PI*x[1]);
  gradQ[1] = -1.1*M_PI*sin(2.1*M_PI*x[0])*sin(1.1*M_PI*x[1])*sin(0.4*M_PI*x[2]);
  gradQ[2] = 0.4*M_PI*sin(2.1*M_PI*x[0])*cos(1.1*M_PI*x[1])*cos(0.4*M_PI*x[2]);
  PetscFunctionReturn(0);
}

PetscErrorCode mms_evaluate_F_ex1(const PetscReal x[],PetscReal F[])
{
  F[0] = -2.1*M_PI*sin(0.4*M_PI*x[2])*cos(2.1*M_PI*x[0])*cos(1.1*M_PI*x[1]);
  F[1] = 1.1*M_PI*sin(2.1*M_PI*x[0])*sin(1.1*M_PI*x[1])*sin(0.4*M_PI*x[2]);
  F[2] = -0.4*M_PI*sin(2.1*M_PI*x[0])*cos(1.1*M_PI*x[1])*cos(0.4*M_PI*x[2]);
  PetscFunctionReturn(0);
}

PetscErrorCode mms_evaluate_f_ex1(const PetscReal x[],PetscReal f[])
{
  f[0] = -5.78*pow(M_PI, 2)*sin(2.1*M_PI*x[0])*sin(0.4*M_PI*x[2])*cos(1.1*M_PI*x[1]);
  PetscFunctionReturn(0);
}




PetscErrorCode bcset_dirichlet_mms(
                                   FVDA fv,DACellFace face,PetscInt nfaces,
                                   const PetscReal coor[],const PetscReal normal[],const PetscInt cell[],PetscReal time,
                                   FVFluxType flux_type[],PetscReal bcvalue[],
                                   void *ctx)
{
  PetscInt f;
  PetscErrorCode ierr;
  for (f=0; f<nfaces; f++) {
    flux_type[f] = FVFLUX_DIRICHLET_CONSTRAINT;
    ierr = mms_evaluate_Q_ex1(&coor[3*f],&bcvalue[f]);CHKERRQ(ierr);
  }
  PetscFunctionReturn(0);
}

/*
 Note the neagtive sign in the Neuammn condition
 
 Our FV implementation uses this
 \int_S k grad(Q) .n dS
 
 The weak form of -div(-k grad(Q)) = f is
 + \int_V grad(w) . (-k grad(Q)) dV - \int_S w (-k grad(Q)) . n dS = \int_V f dV
 + \int_V grad(w) . (-k grad(Q)) dV + \int_S w k grad(Q) . n dS = \int_V f dV
 
 Our MMS function computes the flux as -k grad(Q), but since our implementation absorbs
 the two negatives, we need to flip the sign of the flux returned from the MMS solution.
 */
PetscErrorCode bcset_neumann_mms(
                                 FVDA fv,DACellFace face,PetscInt nfaces,
                                 const PetscReal coor[],const PetscReal normal[],const PetscInt cell[],PetscReal time,
                                 FVFluxType flux_type[],PetscReal bcvalue[],
                                 void *ctx)
{
  PetscInt f;
  PetscReal flux[3];
  PetscErrorCode ierr;
  for (f=0; f<nfaces; f++) {
    const PetscReal *x = &coor[3*f];
    const PetscReal *n = &normal[3*f];
    
    flux_type[f] = FVFLUX_NEUMANN_CONSTRAINT;
    ierr = mms_evaluate_F_ex1(x,flux);CHKERRQ(ierr);
    bcvalue[f] = -(flux[0] * n[0] + flux[1] * n[1] + flux[2] * n[2]);
  }
  PetscFunctionReturn(0);
}

PetscErrorCode EvaluateRHS_mms(FVDA fv,PetscReal time,Vec F)
{
  PetscErrorCode  ierr;
  PetscReal       cell_x[3],f_mms,dV;
  const PetscInt  NSD = 3;
  PetscReal       cell_coor[3 * DACELL3D_Q1_SIZE];
  Vec             coorl;
  const PetscReal *_geom_coor;
  PetscReal       *_F;
  PetscInt        i,d,c,dm_nel,dm_nen;
  const PetscInt  *dm_element,*element;
  
  ierr = VecZeroEntries(F);CHKERRQ(ierr);
  ierr = VecGetArray(F,&_F);CHKERRQ(ierr);
  
  ierr = DMDAGetElements(fv->dm_geometry,&dm_nel,&dm_nen,&dm_element);CHKERRQ(ierr);
  ierr = DMCreateLocalVector(fv->dm_geometry,&coorl);CHKERRQ(ierr);
  ierr = DMGlobalToLocal(fv->dm_geometry,fv->vertex_coor_geometry,INSERT_VALUES,coorl);CHKERRQ(ierr);
  ierr = VecGetArrayRead(coorl,&_geom_coor);CHKERRQ(ierr);
  
  for (c=0; c<fv->ncells; c++) {
    element = (const PetscInt*)&dm_element[DACELL3D_Q1_SIZE * c];
    
    ierr = DACellGeometry3d_GetCoordinates(element,_geom_coor,cell_coor);CHKERRQ(ierr);
    _EvaluateCellVolume3d(cell_coor,&dV);
    
    /* average cell vertex coordinates */
    for (d=0; d<NSD; d++) {
      cell_x[d] = 0.0;
      for (i=0; i<DACELL3D_Q1_SIZE; i++) {
        cell_x[d] += cell_coor[NSD * i + d];
      }
      cell_x[d] /= (PetscReal)DACELL3D_Q1_SIZE;
    }
    
    ierr = mms_evaluate_f_ex1(cell_x,&f_mms);CHKERRQ(ierr);
    _F[c] = f_mms * dV;
  }
  ierr = VecRestoreArray(F,&_F);CHKERRQ(ierr);
  ierr = VecRestoreArrayRead(coorl,&_geom_coor);CHKERRQ(ierr);
  ierr = VecDestroy(&coorl);CHKERRQ(ierr);
  
  PetscFunctionReturn(0);
}

PetscErrorCode create_level(PetscInt m[],FVDA *_fv)
{
  PetscErrorCode ierr;
  PetscInt       bc = 0;
  FVDA           fv;

  
  ierr = PetscOptionsGetInt(NULL,NULL,"-bc",&bc,NULL);CHKERRQ(ierr);
  
  ierr = FVDACreate(PETSC_COMM_WORLD,&fv);CHKERRQ(ierr);
  ierr = FVDASetDimension(fv,3);CHKERRQ(ierr);
  ierr = FVDASetSizes(fv,NULL,m);CHKERRQ(ierr);
  ierr = FVDASetProblemType(fv,PETSC_FALSE,FVDA_ELLIPTIC,0,0);CHKERRQ(ierr);
  ierr = FVDASetUp(fv);CHKERRQ(ierr);
  {
    Vec gcoor;
    PetscReal omega[] = {1.3, 1.1};
    
    ierr = DMDASetUniformCoordinates(fv->dm_geometry,-1.0,1.0,-1.0,1.0,-1.0,1.0);CHKERRQ(ierr);
    ierr = DMDAWarpCoordinates_SinJMax(fv->dm_geometry,0.01,omega);CHKERRQ(ierr);
    
    ierr = DMGetCoordinates(fv->dm_geometry,&gcoor);CHKERRQ(ierr);
    ierr = VecCopy(gcoor,fv->vertex_coor_geometry);CHKERRQ(ierr);
  }
  ierr = FVDAUpdateGeometry(fv);CHKERRQ(ierr);
  
  ierr = FVDARegisterFaceProperty(fv,"k",1);CHKERRQ(ierr);
  {
    PetscInt  f,nfaces;
    PetscReal *k;
    
    ierr = FVDAGetFaceInfo(fv,&nfaces,NULL,NULL,NULL,NULL);CHKERRQ(ierr);
    ierr = FVDAGetFacePropertyArray(fv,0,&k);CHKERRQ(ierr);
    for (f=0; f<nfaces; f++) {
      k[f] = 1.0;
    }
  }
  
  switch (bc) {
    case 0:
      ierr = FVDAFaceIterator(fv,DACELL_FACE_W,PETSC_FALSE,0.0,bcset_dirichlet_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_E,PETSC_FALSE,0.0,bcset_dirichlet_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_N,PETSC_FALSE,0.0,bcset_dirichlet_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_S,PETSC_FALSE,0.0,bcset_dirichlet_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_F,PETSC_FALSE,0.0,bcset_dirichlet_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_B,PETSC_FALSE,0.0,bcset_dirichlet_mms,NULL);CHKERRQ(ierr);
      break;
      
    case 1:
      ierr = FVDAFaceIterator(fv,DACELL_FACE_N,PETSC_FALSE,0.0,bcset_dirichlet_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_S,PETSC_FALSE,0.0,bcset_dirichlet_mms,NULL);CHKERRQ(ierr);
      
      ierr = FVDAFaceIterator(fv,DACELL_FACE_W,PETSC_TRUE,0.0,bcset_neumann_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_E,PETSC_TRUE,0.0,bcset_neumann_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_F,PETSC_TRUE,0.0,bcset_neumann_mms,NULL);CHKERRQ(ierr);
      ierr = FVDAFaceIterator(fv,DACELL_FACE_B,PETSC_TRUE,0.0,bcset_neumann_mms,NULL);CHKERRQ(ierr);
      break;
      
    default:
      SETERRQ(PETSC_COMM_WORLD,PETSC_ERR_SUP,"Valid choices for -bc are {0,1}");
      break;
  }
  
  *_fv = fv;
  
  PetscFunctionReturn(0);
}

PetscErrorCode eval_F_fas(SNES snes,Vec X,Vec F,void *data)
{
  PetscErrorCode    ierr;
  DM                dm;
  FVDA              fv = NULL;

  ierr = SNESGetApplicationContext(snes,(void*)&fv);CHKERRQ(ierr);
  ierr = SNESGetDM(snes,&dm);CHKERRQ(ierr);
  printf("fv %p\n",fv);
  printf("dm %p\n",dm);
  
  PetscFunctionReturn(0);
}

PetscErrorCode t3_warp_mms(void)
{
  PetscErrorCode ierr;
  PetscInt       mx = 12, nlevels = 2, k;
  PetscInt       m[] = {mx,mx,mx};
  FVDA           fv,fvl[20];
  Vec            X,F,rhs;
  Mat            J;
  DM             dm,dmc[20];
  SNES           snes;
  PetscBool      found = PETSC_FALSE;
  
  ierr = PetscOptionsGetInt(NULL,NULL,"-mx",&mx,&found);CHKERRQ(ierr);
  if (found) {
    m[0] = mx;
    m[1] = mx;
    m[2] = mx;
  }

  ierr = PetscOptionsGetInt(NULL,NULL,"-nlvl",&nlevels,NULL);CHKERRQ(ierr);

  ierr = create_level(m,&fvl[0]);CHKERRQ(ierr);
  printf("fvl[0] %p\n",fvl[0]);
  for (k=1; k<nlevels; k++) {
    m[0] *= 2;
    m[1] *= 2;
    m[2] *= 2;
    ierr = create_level(m,&fvl[k]);CHKERRQ(ierr);
    printf("  fvl[%d] %p\n",(int)k,fvl[k]);
  }
  
  fv = fvl[nlevels-1];
 
  for (k=0; k<nlevels; k++) {
    ierr = DMClone(fvl[k]->dm_fv,&dmc[k]);CHKERRQ(ierr);
    ierr = DMDASetInterpolationType(dmc[k],DMDA_Q0);CHKERRQ(ierr);
    printf("dmc[%d] %p\n",(int)k,dmc[k]);
  }

  /*
  for (k=0; k<nlevels; k++) {
    PetscContainer c;
    
    
  }
  */
  
  dm = fv->dm_fv;
  ierr = DMCreateMatrix(dm,&J);CHKERRQ(ierr);
  ierr = DMCreateGlobalVector(dm,&X);CHKERRQ(ierr);
  ierr = DMCreateGlobalVector(dm,&F);CHKERRQ(ierr);
  ierr = DMCreateGlobalVector(dm,&rhs);CHKERRQ(ierr);
  
  ierr = SNESCreate(PETSC_COMM_WORLD,&snes);CHKERRQ(ierr);
  ierr = SNESSetDM(snes,dmc[nlevels-1]);CHKERRQ(ierr);
  ierr = SNESSetSolution(snes,X);CHKERRQ(ierr);
  ierr = SNESSetApplicationContext(snes,(void*)fv);CHKERRQ(ierr);
  
  ierr = EvaluateRHS_mms(fv,0.0,rhs);CHKERRQ(ierr);

  //ierr = SNESSetFunction(snes,F,eval_F,NULL);CHKERRQ(ierr);
  //ierr = SNESSetJacobian(snes,J,J,eval_J,NULL);CHKERRQ(ierr);

  //ierr = SNESSetFunction(snes,F,eval_F_hr,NULL);CHKERRQ(ierr);
  //ierr = SNESSetJacobian(snes,J,J,SNESComputeJacobianDefaultColor,NULL);CHKERRQ(ierr);

  ierr = SNESSetFunction(snes,F,eval_F_hr,NULL);CHKERRQ(ierr);
  ierr = SNESSetJacobian(snes,J,J,eval_J,NULL);CHKERRQ(ierr);


  ierr = SNESSetFromOptions(snes);CHKERRQ(ierr);
  printf("snes %p \n",snes);

  
  //ierr = SNESSetType(snes,SNESFAS);CHKERRQ(ierr);
  ierr = SNESFASSetLevels(snes,nlevels,NULL);CHKERRQ(ierr);
  SNESSetUp(snes);
  for (k=0; k<nlevels; k++) {
    SNES lsnes,smooth;
    ierr = SNESFASGetCycleSNES(snes,k,&lsnes);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(lsnes,(void*)fvl[k]);CHKERRQ(ierr);
    printf("lsnes %p \n",lsnes);
    ierr = SNESFASCycleGetSmootherDown(lsnes, &smooth);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(smooth,(void*)fvl[k]);CHKERRQ(ierr);
    ierr = SNESFASCycleGetSmootherUp(lsnes, &smooth);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(smooth,(void*)fvl[k]);CHKERRQ(ierr);
    ierr = SNESFASCycleGetSmoother(lsnes, &smooth);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(smooth,(void*)fvl[k]);CHKERRQ(ierr);
  }
  {
    SNES lsnes;
    ierr = SNESFASGetCoarseSolve(snes,&lsnes);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(lsnes,(void*)fvl[0]);CHKERRQ(ierr);
    printf("lsnes %p [coarse]\n",lsnes);
  }
  
  ierr = SNESSolve(snes,rhs,X);CHKERRQ(ierr);
  
  ierr = MatDestroy(&J);CHKERRQ(ierr);
  ierr = VecDestroy(&rhs);CHKERRQ(ierr);
  ierr = VecDestroy(&F);CHKERRQ(ierr);
  ierr = VecDestroy(&X);CHKERRQ(ierr);
  ierr = SNESDestroy(&snes);CHKERRQ(ierr);
  //ierr = FVDADestroy(&fv);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}


/*

 FAS setup designed to work with arbitrary refine/coarsening factors.
 
 However, there is missing support for coarsening with factors other than 1 or 2 for DA(2D,3D)
 
 [0]PETSC ERROR: --------------------- Error Message --------------------------------------------------------------
 [0]PETSC ERROR: Invalid argument
 [0]PETSC ERROR: Coarsening factor in x must be 1 or 2
 [0]PETSC ERROR: See https://www.mcs.anl.gov/petsc/documentation/faq.html for trouble shooting.
 [0]PETSC ERROR: Petsc Release Version 3.13.3, Jul 01, 2020
 [0]PETSC ERROR: ./arch-gnu-c-debug/bin/fv-ex3-fas.app on a arch-gnu-c-debug named dhcp211.geophysik.uni-muenchen.de by dmay Thu Aug 13 11:26:18 2020
 [0]PETSC ERROR: Configure options --download-mpich=yes --download-suitesparse=yes
 [0]PETSC ERROR: #1 DMCreateInterpolation_DA_3D_Q0() line 662 in /Users/dmay/software/petsc-3.13.3/src/dm/impls/da/dainterp.c
 [0]PETSC ERROR: #2 DMCreateInterpolation_DA() line 1104 in /Users/dmay/software/petsc-3.13.3/src/dm/impls/da/dainterp.c
 [0]PETSC ERROR: #3 DMCreateInterpolation() line 1141 in /Users/dmay/software/petsc-3.13.3/src/dm/interface/dm.c
 [0]PETSC ERROR: #4 SNESSetUp_FAS() line 123 in /Users/dmay/software/petsc-3.13.3/src/snes/impls/fas/fas.c
 [0]PETSC ERROR: #5 SNESSetUp() line 3147 in /Users/dmay/software/petsc-3.13.3/src/snes/interface/snes.c

*/
PetscErrorCode t3_warp_mms_r(void)
{
  PetscErrorCode ierr;
  PetscInt       mx = 12, nlevels = 2, k, rfac = 2;
  PetscInt       m[] = {mx,mx,mx};
  FVDA           fv,fvl[20];
  Vec            X,F,rhs;
  Mat            J;
  DM             dm,dmc[20];
  SNES           snes;
  PetscBool      found = PETSC_FALSE;

  ierr = PetscOptionsGetInt(NULL,NULL,"-rfac",&rfac,NULL);CHKERRQ(ierr);

  ierr = PetscOptionsGetInt(NULL,NULL,"-mx",&mx,&found);CHKERRQ(ierr);
  if (found) {
    m[0] = mx;
    m[1] = mx;
    m[2] = mx;
  }
  
  ierr = PetscOptionsGetInt(NULL,NULL,"-nlvl",&nlevels,NULL);CHKERRQ(ierr);
  
  ierr = create_level(m,&fvl[0]);CHKERRQ(ierr);
  ierr = DMDASetRefinementFactor(fvl[0]->dm_fv,rfac,rfac,rfac);CHKERRQ(ierr);
  
  printf("fvl[0] %p\n",fvl[0]);
  for (k=1; k<nlevels; k++) {
    m[0] *= rfac;
    m[1] *= rfac;
    m[2] *= rfac;
    ierr = create_level(m,&fvl[k]);CHKERRQ(ierr);
    ierr = DMDASetRefinementFactor(fvl[k]->dm_fv,rfac,rfac,rfac);CHKERRQ(ierr);
    printf("  fvl[%d] %p\n",(int)k,fvl[k]);
  }
  
  fv = fvl[nlevels-1];

  /* clone and then refine, otherwise dd->coarse_{x,y,z} never gets set */
  /*
  for (k=0; k<nlevels; k++) {
    ierr = DMClone(fvl[k]->dm_fv,&dmc[k]);CHKERRQ(ierr);
    ierr = DMDASetInterpolationType(dmc[k],DMDA_Q0);CHKERRQ(ierr);
    ierr = DMDASetRefinementFactor(dmc[k],8,8,8);CHKERRQ(ierr);
    printf("dmc[%d] %p\n",k,dmc[k]);
  }
  */

  ierr = DMClone(fvl[0]->dm_fv,&dmc[0]);CHKERRQ(ierr);
  ierr = DMDASetInterpolationType(dmc[0],DMDA_Q0);CHKERRQ(ierr);
  ierr = DMDASetRefinementFactor(dmc[0],rfac,rfac,rfac);CHKERRQ(ierr);
  printf("dmc[0] %p\n",dmc[0]);
  for (k=1; k<nlevels; k++) {
    ierr = DMRefine(dmc[k-1],PETSC_COMM_WORLD,&dmc[k]);CHKERRQ(ierr);
    printf("dmc[%d] %p\n",(int)k,dmc[k]);
    ierr = DMDASetInterpolationType(dmc[k],DMDA_Q0);CHKERRQ(ierr);
    ierr = DMDASetRefinementFactor(dmc[k],rfac,rfac,rfac);CHKERRQ(ierr);
  }

  dm = fv->dm_fv;
  ierr = DMCreateMatrix(dm,&J);CHKERRQ(ierr);
  ierr = DMCreateGlobalVector(dm,&X);CHKERRQ(ierr);
  ierr = DMCreateGlobalVector(dm,&F);CHKERRQ(ierr);
  ierr = DMCreateGlobalVector(dm,&rhs);CHKERRQ(ierr);
  
  ierr = SNESCreate(PETSC_COMM_WORLD,&snes);CHKERRQ(ierr);
  ierr = SNESSetDM(snes,dmc[nlevels-1]);CHKERRQ(ierr);
  ierr = SNESSetSolution(snes,X);CHKERRQ(ierr);
  ierr = SNESSetApplicationContext(snes,(void*)fv);CHKERRQ(ierr);
  
  ierr = EvaluateRHS_mms(fv,0.0,rhs);CHKERRQ(ierr);
  
  //ierr = SNESSetFunction(snes,F,eval_F,NULL);CHKERRQ(ierr);
  //ierr = SNESSetJacobian(snes,J,J,eval_J,NULL);CHKERRQ(ierr);
  
  //ierr = SNESSetFunction(snes,F,eval_F_hr,NULL);CHKERRQ(ierr);
  //ierr = SNESSetJacobian(snes,J,J,SNESComputeJacobianDefaultColor,NULL);CHKERRQ(ierr);
  
  ierr = SNESSetFunction(snes,F,eval_F_hr,NULL);CHKERRQ(ierr);
  ierr = SNESSetJacobian(snes,J,J,eval_J,NULL);CHKERRQ(ierr);
  
  
  ierr = SNESSetFromOptions(snes);CHKERRQ(ierr);
  printf("snes %p \n",snes);
  
  
  //ierr = SNESSetType(snes,SNESFAS);CHKERRQ(ierr);
  ierr = SNESFASSetLevels(snes,nlevels,NULL);CHKERRQ(ierr);
  SNESSetUp(snes);
  for (k=0; k<nlevels; k++) {
    SNES lsnes,smooth;
    ierr = SNESFASGetCycleSNES(snes,k,&lsnes);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(lsnes,(void*)fvl[k]);CHKERRQ(ierr);
    printf("lsnes %p \n",lsnes);
    ierr = SNESFASCycleGetSmootherDown(lsnes, &smooth);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(smooth,(void*)fvl[k]);CHKERRQ(ierr);
    ierr = SNESFASCycleGetSmootherUp(lsnes, &smooth);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(smooth,(void*)fvl[k]);CHKERRQ(ierr);
    ierr = SNESFASCycleGetSmoother(lsnes, &smooth);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(smooth,(void*)fvl[k]);CHKERRQ(ierr);
  }
  {
    SNES lsnes;
    ierr = SNESFASGetCoarseSolve(snes,&lsnes);CHKERRQ(ierr);
    ierr = SNESSetApplicationContext(lsnes,(void*)fvl[0]);CHKERRQ(ierr);
    printf("lsnes %p [coarse]\n",lsnes);
  }
  
  ierr = SNESSolve(snes,rhs,X);CHKERRQ(ierr);
  
  ierr = MatDestroy(&J);CHKERRQ(ierr);
  ierr = VecDestroy(&rhs);CHKERRQ(ierr);
  ierr = VecDestroy(&F);CHKERRQ(ierr);
  ierr = VecDestroy(&X);CHKERRQ(ierr);
  ierr = SNESDestroy(&snes);CHKERRQ(ierr);
  //ierr = FVDADestroy(&fv);CHKERRQ(ierr);
  PetscFunctionReturn(0);
}


/*
 
 t3_warp_mms()
 
 ./arch-gnu-c-debug/bin/fv-ex3-fas.app -snes_monitor -snes_viewx -fas_coarse_pc_type lu -fas_coarse_snes_monitor -snes_type fas -fas_levels_snes_type ngmres -fas_levels_snes_max_it 4 -nlvl 4 -mx 3 -warp_sin_amp 0.5
 
 0 SNES Function norm 6.553419489795e+00
   0 SNES Function norm 2.475068880382e-01
   1 SNES Function norm 9.322856742774e-03
   2 SNES Function norm 8.257293371811e-05
   3 SNES Function norm 1.724036169944e-06
   4 SNES Function norm 1.727844915249e-08
   5 SNES Function norm 2.761071252077e-10
 1 SNES Function norm 4.435888444296e-01
   0 SNES Function norm 4.380799392080e-02
   1 SNES Function norm 1.115488613110e-03
   2 SNES Function norm 1.035310694523e-05
   3 SNES Function norm 2.003384955529e-07
   4 SNES Function norm 1.696865249343e-09
   5 SNES Function norm 3.293461606883e-11
 2 SNES Function norm 5.030573421257e-02
   0 SNES Function norm 6.076735582926e-03
   1 SNES Function norm 1.310804205163e-04
   2 SNES Function norm 1.216519004946e-06
   3 SNES Function norm 2.472568853936e-08
   4 SNES Function norm 2.651488001346e-10
 3 SNES Function norm 1.410188898219e-02
   0 SNES Function norm 1.183518169518e-03
   1 SNES Function norm 4.040497098015e-05
   2 SNES Function norm 3.834093513589e-07
   3 SNES Function norm 7.636564351564e-09
   4 SNES Function norm 7.034290751968e-11
 4 SNES Function norm 2.488489147477e-03
   0 SNES Function norm 3.063840715673e-04
   1 SNES Function norm 5.968620054062e-06
   2 SNES Function norm 5.504226454053e-08
   3 SNES Function norm 1.086984764321e-09
   4 SNES Function norm 1.326073693958e-11
 5 SNES Function norm 6.639347583885e-04
   0 SNES Function norm 5.105122203530e-05
   1 SNES Function norm 1.724828571109e-06
   2 SNES Function norm 1.627804639067e-08
   3 SNES Function norm 3.173133787389e-10
 6 SNES Function norm 1.305971903854e-04
   0 SNES Function norm 1.116524145487e-05
   1 SNES Function norm 2.723614513505e-07
   2 SNES Function norm 2.424966867768e-09
   3 SNES Function norm 4.575018579112e-11
 7 SNES Function norm 3.519796771182e-05
   0 SNES Function norm 1.674677467348e-06
   1 SNES Function norm 6.316113427532e-08
   2 SNES Function norm 5.785397791905e-10
   3 SNES Function norm 1.037806511215e-11
 8 SNES Function norm 7.012954572786e-06
   0 SNES Function norm 3.666321636983e-07
   1 SNES Function norm 1.198139301194e-08
   2 SNES Function norm 1.044487691527e-10
 9 SNES Function norm 1.801308295473e-06
   0 SNES Function norm 1.134123803132e-07
   1 SNES Function norm 2.861847510458e-09
   2 SNES Function norm 2.493979210109e-11
 10 SNES Function norm 3.774340338746e-07
   0 SNES Function norm 1.608779770313e-08
   1 SNES Function norm 6.353691045629e-10
 11 SNES Function norm 9.506665529165e-08
   0 SNES Function norm 5.360800738789e-09
   1 SNES Function norm 1.375144774538e-10
 12 SNES Function norm 2.141707327093e-08

 
 compare with
 
 ./arch-gnu-c-debug/bin/fv-ex3.app -tid 2 -snes_monitor -snes_viewx -pc_type lu  -nlvl 4 -mx 24 -warp_sin_amp 0.5
 
 which doesn't converge fast at all
 
 0 SNES Function norm 6.553419489795e+00
 1 SNES Function norm 4.842342284840e-01
 2 SNES Function norm 2.198661569802e-01
 3 SNES Function norm 1.627864858794e-01
 4 SNES Function norm 1.167677080776e-01
 5 SNES Function norm 8.458379873836e-02
 6 SNES Function norm 6.158434167902e-02
 7 SNES Function norm 4.520227982529e-02
 8 SNES Function norm 3.307732830687e-02
 9 SNES Function norm 2.413267765013e-02
 10 SNES Function norm 1.756117940498e-02
 11 SNES Function norm 1.274586007357e-02
 12 SNES Function norm 9.221911517551e-03
 13 SNES Function norm 6.649900494861e-03
 14 SNES Function norm 4.778675860219e-03
 ... I gave up waiting ...
 
*/
extern PetscErrorCode PCCreate_DMDARepart(PC pc);
int main(int argc,char **args)
{
  PetscErrorCode ierr;
  PetscInt       tid = 2;
  
  ierr = PetscInitialize(&argc,&args,(char*)0,NULL);if (ierr) return ierr;
  ierr = PCRegister("dmdarepart",PCCreate_DMDARepart);CHKERRQ(ierr);
  ierr = PetscOptionsGetInt(NULL,NULL,"-tid",&tid,NULL);CHKERRQ(ierr);
  ierr = PetscOptionsGetBool(NULL,NULL,"-operator_fvspace",&operator_fvspace,NULL);CHKERRQ(ierr);
  ierr = PetscOptionsGetBool(NULL,NULL,"-view",&view,NULL);CHKERRQ(ierr);
  switch (tid) {
    case 2:
      ierr = t3_warp_mms();CHKERRQ(ierr);
      //ierr = t3_warp_mms_r();CHKERRQ(ierr);
      break;
    default: SETERRQ(PETSC_COMM_WORLD,PETSC_ERR_USER,"Valid values for -tid {0,1,2}");
      break;
  }
  ierr = PetscFinalize();
  return ierr;
}
